package rbacrpcserver

import (
	"context"

	"github.com/twitchtv/twirp"

	"code.justin.tv/devrel/devsite-rbac/backend/companyresources"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

func (s *Server) CreateResource(ctx context.Context, params *rbacrpc.CreateResourceRequest) (*rbacrpc.CreateResourceResponse, error) {
	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	if err := validateCreateResource(params); err != nil {
		return nil, err
	}

	if err := s.createResource(ctx, params); err != nil {
		return nil, err
	}

	if err := s.Backend.Commit(ctx); err != nil {
		return nil, err
	}

	return &rbacrpc.CreateResourceResponse{}, nil
}

func validateCreateResource(params *rbacrpc.CreateResourceRequest) error {
	if params.Resource == nil {
		return twirp.RequiredArgumentError("resource")
	}

	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"company_id", params.CompanyId},
		{"resource.type", params.Resource.Type},
		{"resource.external_id", params.Resource.ExternalId},
	}); err != nil {
		return err
	}

	if err := errorutil.ValidateUUID("company_id", params.CompanyId); err != nil {
		return err
	}

	return nil
}

func (s *Server) createResource(ctx context.Context, params *rbacrpc.CreateResourceRequest) error {
	company, err := s.GetCompany(ctx, &rbacrpc.Id{Id: params.CompanyId})
	if err != nil {
		return err
	}

	// Validate resource isn't already owned
	if err := s.verifyResourceNotAlreadyOwned(ctx, params.Resource.Type, params.Resource.ExternalId); err != nil {
		return err
	}

	// Add resource to company
	return s.Backend.InsertResource(ctx, &companyresources.ResourceType{
		Type:       params.Resource.Type,
		ExternalId: params.Resource.ExternalId,
		CompanyID:  company.Id,
	})
}
