package rbacrpcserver

import (
	"context"
	"fmt"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/clients/dart"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/models"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

func (s *Server) DeleteCompanyApplication(ctx context.Context, params *rbacrpc.DeleteCompanyApplicationRequest) (*rbacrpc.Empty, error) {
	capp, err := s.Backend.GetCompanyApplication(ctx, params.Id)
	if err != nil {
		return nil, err
	}
	cappRPC := capp.ToRPC()

	games := make([]string, len(cappRPC.Games))
	for i, game := range cappRPC.Games {
		games[i] = game.Id
	}

	// Update status to rejected
	err = s.Backend.UpdateCompanyApplicationStatus(ctx, cappRPC.Id, models.CompanyApplicationRejected)
	if err != nil {
		return nil, err
	}

	s.sendCompanyApplicationRejectionEmail(ctx, params, cappRPC)
	return &rbacrpc.Empty{}, nil
}

func (s *Server) sendCompanyApplicationRejectionEmail(ctx context.Context, params *rbacrpc.DeleteCompanyApplicationRequest, capp *rbacrpc.CompanyApplication) {
	if params.SkipEmail {
		return
	}

	async(ctx, func(ctx context.Context) error {
		// Send rejected email via DART integration
		emailTraceId, emailError := s.Dart.SendDeveloperCompanyActionDenied(ctx, capp.TwitchId, capp.CompanyName) // if email was sent via dart we can track the status of that message with this

		s.auditEmailAction(ctx, DartEmailAction{
			UserTwitchID:   auth.GetTwitchID(ctx),
			EntityTwitchID: capp.TwitchId,
			ActionFormat:   fmt.Sprintf("Email: Sent type %s with traceid of %s", dart.DeveloperCompanyActionDenied, emailTraceId),
			CompanyID:      capp.Id,
		})

		return emailError
	}, logx.Fields{
		"companyApplicationID": params.Id,
	})
}
