package rbacrpcserver

import (
	"context"

	"code.justin.tv/devrel/devsite-rbac/backend/companyresources"
	"code.justin.tv/devrel/devsite-rbac/clients/owlcli"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/models/permissions"
	"github.com/twitchtv/twirp"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	owl "code.justin.tv/web/owl/client"
)

func (s *Server) DetachExtensionFromCompany(ctx context.Context, params *rbacrpc.DetachExtensionRequest) (*rbacrpc.Empty, error) {
	requestingTwitchID := auth.GetTwitchID(ctx)
	if requestingTwitchID == "" {
		return nil, twirp.NewError(twirp.PermissionDenied, "OAuth token required")
	}

	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"company_id", params.CompanyId},
		{"extension_id", params.ExtensionId},
	}); err != nil {
		return nil, err
	}
	if err := errorutil.ValidateUUID("company_id", params.CompanyId); err != nil {
		return nil, err
	}

	// DB transaction
	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	// Check if company exists
	_, err := s.Backend.SelectCompany(ctx, params.CompanyId)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.InvalidArgumentError("company_id", "is not a valid company id")
	}
	if err != nil {
		return nil, err
	}

	// Validate new owner_id, only used on monetized extensions, needs to have monetization enabled
	if params.OwnerId != "" {
		isValidID, err := s.Users.IsValidTwitchID(ctx, params.OwnerId)
		if err != nil {
			return nil, err // users service is down
		}
		if !isValidID {
			return nil, twirp.InvalidArgumentError("owner_id", "is invalid or doesn't exist")
		}
		err = s.validateCanBeBillingManager(ctx, params.OwnerId)
		if err != nil {
			return nil, err // FailedPrecondition, or PermissionDenied or Internal
		}
	}

	// Delete extension resource in RBAC
	err = s.Backend.DeleteResource(ctx, &companyresources.ResourceType{
		ExternalId: params.ExtensionId,
		Type:       permissions.Extension,
		CompanyID:  params.CompanyId,
	})
	if err != nil {
		return nil, err
	}

	// Load previous Owl client values to allow rollback
	prevClient, err := s.Owl.GetClient(ctx, params.ExtensionId)
	if errorutil.Is(err, owl.ErrInvalidClientID) {
		return nil, twirp.InvalidArgumentError("extension_id", "is not a valid client id")
	}
	if err != nil {
		return nil, err
	}

	// Update client in Owl
	err = s.Owl.UpdateClientOwner(ctx, owlcli.UpdateClientRequest{
		ClientID:           params.ExtensionId,
		OwnerID:            params.OwnerId,
		GroupID:            owlcli.ResetGroupID,
		RequestingTwitchID: requestingTwitchID,
	})
	if errorutil.Is(err, owl.ErrInvalidClientID) {
		return nil, twirp.InvalidArgumentError("extension_id", "is not a valid client id")
	}
	if errorutil.Is(err, owl.ErrInvalidUserID) {
		return nil, twirp.InvalidArgumentError("owner_id", "is not a valid owner id")
	}
	if err != nil {
		return nil, err
	}

	// DB Transaction commit
	if err = s.Backend.Commit(ctx); err != nil {
		s.rollbackOwlClient(ctx, prevClient, requestingTwitchID)
		return nil, err
	}

	return &rbacrpc.Empty{}, nil
}
