package rbacrpcserver

import (
	"context"

	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/twitchtv/twirp"
)

func (s *Server) GetUsersByCompanyId(ctx context.Context, params *rbacrpc.GetUsersByCompanyIdRequest) (*rbacrpc.GetUsersByCompanyIdResponse, error) {
	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"company_id", params.Id},
	}); err != nil {
		return nil, err
	}
	if err := errorutil.ValidateUUID("company_id", params.Id); err != nil {
		return nil, err
	}

	// validate and format role parameter
	if params.Role != "" {
		formattedRole, err := SanitizeRole("role", params.Role)
		if err != nil {
			return nil, err
		}
		params.Role = formattedRole
	}

	// check if company exists
	_, err := s.Backend.SelectCompany(ctx, params.Id)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.NotFoundError("company not found")
	}
	if err != nil {
		return nil, err
	}

	if params.Limit == 0 {
		params.Limit = uint64(10)
	}

	list, total, err := s.Memberships.ListMemberships(ctx, memberships.ListMembershipsParams{
		CompanyID: params.Id,
		Role:      params.Role,

		Limit:      params.Limit,
		Offset:     params.Offset,
		OrderBy:    params.SortBy,
		OrderByDir: params.OrderBy.String(),
	})
	if err != nil {
		return nil, err
	}

	return &rbacrpc.GetUsersByCompanyIdResponse{
		Id:          params.Id,
		Memberships: memberships.ListToRPC(list),
		Total:       uint64(total),
		Offset:      params.Offset,
	}, nil
}
