package rbacrpcserver

import (
	"context"
	"fmt"
	"strconv"
	"strings"

	"code.justin.tv/chat/golibs/logx"

	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/companies"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/clients/dart"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/models"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"

	"github.com/twitchtv/twirp"
)

func (s *Server) OnboardCompany(ctx context.Context, params *rbacrpc.OnboardCompanyRequest) (*rbacrpc.OnboardCompanyResponse, error) {
	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	// Check if company application exists
	companyApp, err := s.Backend.GetCompanyApplication(ctx, params.Id)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.InvalidArgumentError("id", "company application not found")
	}
	if err != nil {
		return nil, err
	}

	// Check if another company with the same name (simplified as an "identifier") already exists
	identifier := common.Identifier(companyApp.CompanyName)
	alreadyExists, err := s.Backend.CompanyExistsWithIdentifier(ctx, identifier)
	if err != nil {
		return nil, err
	}
	if alreadyExists {
		return nil, twirp.NewError(twirp.AlreadyExists, "company identifier already exists")
	}

	// Create company
	company, err := s.Backend.InsertCompany(ctx, &companies.CreateRequest{
		CompanyName:       companyApp.CompanyName,
		Url:               companyApp.CompanyWebsite,
		Type:              companyApp.CompanyType,
		VhsContractSigned: false,
		CampaignsEnabled:  false,
		Identifier:        identifier,
		ApplicationId:     companyApp.Id,
		CurseCompanyId:    -1,
	})
	if err != nil {
		return nil, err
	}
	s.whitelistNewlyAddedCompany(ctx, companyApp.TwitchId, companyApp.Id)

	// Add initial user as Owner
	err = s.Memberships.InsertMembership(ctx, &memberships.Membership{
		CompanyID: company.Id,
		Role:      ownerRole,
		TwitchID:  companyApp.TwitchId,
		FirstName: companyApp.ContactFirstName,
		LastName:  companyApp.ContactLastName,
		DevTitle:  companyApp.ContactTitle,
		DevEmail:  companyApp.ContactEmail,
	})
	if err != nil {
		return nil, err
	}

	// Make Game Applications if the appliaction has game ids
	gameIDs := strings.Split(companyApp.Games, ",")
	createdGameApps := []*rbacrpc.GameApplication{}
	for _, gameIDStr := range gameIDs {
		gameID, err := strconv.Atoi(gameIDStr)
		if err != nil {
			continue // ignored bad ids
		}

		gameApp, err := s.Backend.InsertGameApplication(ctx, &rbacrpc.CreateGameApplicationRequest{
			GameId:    int32(gameID),
			CompanyId: company.Id,
		})
		if err != nil {
			return nil, err
		}
		createdGameApps = append(createdGameApps, gameApp)
	}

	// Approve application status
	err = s.Backend.UpdateCompanyApplicationStatus(ctx, companyApp.Id, models.CompanyApplicationApproved)
	if err != nil {
		return nil, err
	}

	if err := s.Backend.Commit(ctx); err != nil {
		return nil, err
	}

	currentUserTwitchID := auth.GetTwitchID(ctx)

	s.auditUserRoleChange(ctx, UserRoleAction{
		CurrentUserTwitchID: currentUserTwitchID,
		EntityTwitchID:      companyApp.TwitchId,
		ActionFormat:        "Edit: Assigned company role of %s",
		Role:                "Owner",
		CompanyID:           companyApp.Id,
	})

	for _, gameApp := range createdGameApps {
		s.auditCreateGameApplication(ctx, gameApp)
	}

	async(ctx, func(ctx context.Context) error {
		// Send approved email via DART integration
		emailTraceId, emailError := s.Dart.SendDeveloperCompanyActionApproved(ctx, companyApp.TwitchId, company.CompanyName) // if email was sent via dart we can track the status of that message with this

		s.auditEmailAction(ctx, DartEmailAction{
			UserTwitchID:   currentUserTwitchID,
			EntityTwitchID: companyApp.TwitchId,
			ActionFormat:   fmt.Sprintf("Email: Sent type %s with traceid of %s", dart.DeveloperCompanyActionApproved, emailTraceId),
			CompanyID:      company.Id,
		})

		return emailError
	})

	return &rbacrpc.OnboardCompanyResponse{
		Company: company,
		Id:      company.Id,
	}, nil
}

func (s *Server) whitelistNewlyAddedCompany(ctx context.Context, ownerId string, companyId string) {
	// checking if user is whitelisted in `create_org_user_whitelist`
	resp, err := s.IsValueEnabled(ctx, &rbacrpc.IsValueEnabledRequest{
		FeatureGatingKey: models.NewOrgConsoleUserWhitelistKey,
		ValueInString:    ownerId,
	})
	if err == nil && resp.Enabled {
		// if owner id whitelisted, whitelist this orgID
		_, err = s.FeatureGating.UpdateStringsFeatureGate(ctx, models.NewOrgConsoleOrgWhitelistKey, companyId, rbacrpc.UpdateFeatureGatingValueAction_ADD)
	}
	if err != nil {
		logx.Warn(ctx, err)
	}
}
