package rbacrpcserver

import (
	"context"

	"github.com/twitchtv/twirp"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

func (s *Server) OnboardDeveloper(ctx context.Context, params *rbacrpc.OnboardDeveloperRequest) (*rbacrpc.OnboardDeveloperResponse, error) {
	if params.DeveloperApplicationId == "" {
		return nil, twirp.RequiredArgumentError("developer_application_id")
	}

	devApp, err := s.DevApps.GetDeveloperApplication(ctx, params.DeveloperApplicationId)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.NotFoundError("application not found")
	}
	if err != nil {
		return nil, err
	}
	if params.Role == shadowAccountRole {
		return nil, twirp.NewError(twirp.InvalidArgument, "shadow account user cannot be onboarded as normal users.")
	}

	// Create user
	_, err = s.AddUserToCompany(ctx, &rbacrpc.AddUserToCompanyRequest{
		User: &rbacrpc.CreateUserRequest{
			TwitchId:  devApp.TwitchID,
			FirstName: devApp.FirstName,
			LastName:  devApp.LastName,
			Title:     devApp.Title,
			Email:     devApp.Email,
		},
		CompanyId: devApp.CompanyID,
		Role:      params.Role,
	})
	if err != nil {
		return nil, err
	}

	// Delete application
	err = s.DevApps.DeleteDeveloperApplication(ctx, devApp.ID)
	if err != nil {
		// Send the error to Rollbar, but don't cancel the request because this is our problem and not the users
		logx.Error(ctx, err, logx.Fields{
			"message":                  "user was created but dev app wasn't deleted afterwards",
			"developer_application_id": devApp.ID,
			"twitch_id":                devApp.TwitchID,
			"company_id":               devApp.CompanyID,
		})
	}

	return &rbacrpc.OnboardDeveloperResponse{}, nil
}
