package rbacrpcserver

import (
	"context"
	"fmt"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/clients/dart"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"

	"github.com/twitchtv/twirp"

	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"

	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
)

func (s *Server) RemoveCompanyMembership(ctx context.Context, params *rbacrpc.RemoveCompanyMembershipRequest) (*rbacrpc.Empty, error) {
	if err := s.validateRemoveCompanyMembershipParams(ctx, params); err != nil {
		return nil, err
	}

	// DB Transaction
	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	// Get membership data (used for email and to verify role)
	memb, err := s.Memberships.GetMembership(ctx, params.CompanyId, params.TwitchId)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.NotFoundError("twitch_id")
	}
	if err != nil {
		return nil, err
	}

	if err := s.validateRemoveCompanyMembership(ctx, params, memb.Role); err != nil {
		return nil, err
	}

	if err := s.Memberships.DeleteMembership(ctx, memb.CompanyID, memb.TwitchID); err != nil {
		return nil, err
	}

	if err := s.Backend.Commit(ctx); err != nil {
		return nil, err
	}

	if err := s.updateAllowedChattersByCompanyID(ctx, params.CompanyId); err != nil {
		logx.Error(ctx, errx.Wrap(err, "Failed to updateAllowedChattersByCompanyID, companyID: "+params.CompanyId))
	}

	async(ctx, func(ctx context.Context) error {
		company, err := s.Backend.SelectCompany(ctx, params.CompanyId)
		if err != nil {
			return errx.Wrap(err, "failed to find company for removed user email")
		}
		emailTraceId, emailError := s.Dart.SendDeveloperCompanyUserActionRemoved(ctx, memb.TwitchID, company.CompanyName) // if email was sent via dart we can track the status of that message with this

		s.auditEmailAction(ctx, DartEmailAction{
			UserTwitchID:   params.RequestingTwitchId,
			EntityTwitchID: memb.TwitchID,
			ActionFormat:   fmt.Sprintf("Email: Sent type %s with traceid of %s", dart.DeveloperCompanyUserActionRemoved, emailTraceId),
			CompanyID:      company.Id,
		})

		return emailError
	}, logx.Fields{
		"twitch_id":  memb.TwitchID,
		"role":       memb.Role,
		"company_id": memb.CompanyID,
	})

	s.auditUserRoleChange(ctx, UserRoleAction{
		CurrentUserTwitchID: params.RequestingTwitchId,
		EntityTwitchID:      memb.TwitchID,
		ActionFormat:        "Edit: Removed from role %s in company %s",
		Role:                memb.Role,
		CompanyID:           memb.CompanyID,
	})

	return &rbacrpc.Empty{}, nil
}

func (s *Server) validateRemoveCompanyMembershipParams(ctx context.Context, params *rbacrpc.RemoveCompanyMembershipRequest) error {
	if auth.IsWhitelistAdmin(ctx) {
		params.RequestingTwitchId = auth.GetTwitchID(ctx) // take value from OAuth token by default
	}

	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"company_id", params.CompanyId},
		{"twitch_id", params.TwitchId},
		{"requesting_twitch_id", params.RequestingTwitchId},
	}); err != nil {
		return err
	}

	if err := errorutil.ValidateUUID("company_id", params.CompanyId); err != nil {
		return err
	}

	return nil
}

func (s *Server) validateRemoveCompanyMembership(ctx context.Context, params *rbacrpc.RemoveCompanyMembershipRequest, role string) error {
	if auth.IsWhitelistAdmin(ctx) {
		return nil // whitelisted admins can always remove users, even the last one
	}

	// Check if requesting user has permission to remove other users
	if params.RequestingTwitchId != params.TwitchId { // always allow to remove themselves
		validResp, err := s.ValidateByTwitchID(ctx, &rbacrpc.ValidateQuery{
			UserId:       params.RequestingTwitchId,
			ResourceId:   params.CompanyId,
			Permission:   "removeUser",
			ResourceType: "company",
		})
		if err != nil {
			return err
		}
		if !validResp.Valid {
			return twirp.NewError(twirp.PermissionDenied, "cannot remove company membership")
		}
	}

	// Prevent from removing the owner
	if role == ownerRole {
		return twirp.NewError(twirp.PermissionDenied, "owner cannot be removed")
	}

	// Prevent from removing the shadow account user
	if role == shadowAccountRole {
		return twirp.NewError(twirp.PermissionDenied, "Shadow account user cannot be removed from a company.")
	}

	// Make sure the user is not the assigned billing manager of an extension (e.g. "Administrator" or "Billing_Manager" roles)
	isAssignedBIllingManager, err := s.Backend.IsAssignedBillingManager(ctx, params.CompanyId, params.TwitchId)
	if err != nil {
		return err
	}

	if isAssignedBIllingManager {
		return twirp.NewError(twirp.FailedPrecondition, "member is an assigned billing manager of an extension")
	}

	return nil
}
