package rbacrpcserver

import (
	"context"
	"fmt"
	"strconv"

	"code.justin.tv/devrel/devsite-rbac/backend/common"

	"github.com/twitchtv/twirp"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/devrel/devsite-rbac/backend/actionhistories"
	"code.justin.tv/devrel/devsite-rbac/backend/extensionbillingmanagers"
	"code.justin.tv/devrel/devsite-rbac/clients/owlcli"
	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/models/permissions"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"code.justin.tv/web/owl/oauth2"
)

func (s *Server) SetExtensionBillingManager(ctx context.Context, params *rbacrpc.SetExtensionBillingManagerRequest) (*rbacrpc.SetExtensionBillingManagerResponse, error) {
	if err := s.validateSetExtensionBillingManagerParams(ctx, params); err != nil {
		return nil, err
	}

	// Ensure permissions: can assign an extension billing manager
	if !auth.IsWhitelistAdmin(ctx) {
		validResp, err := s.ValidateByTwitchID(ctx, &rbacrpc.ValidateQuery{
			UserId:       params.RequestingTwitchId,
			ResourceId:   params.ExtensionClientId,
			Permission:   permissions.SetBillingManager,
			ResourceType: permissions.Extension,
		})
		if err != nil {
			return nil, err
		}
		if !validResp.Valid {
			return nil, twirp.NewError(twirp.PermissionDenied, "cannot set billing manager for extension")
		}
	}

	// DB transaction
	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	// Find the companyID that has this extension resource
	companies, err := s.Backend.FindCompaniesWithResource(ctx, params.ExtensionClientId, permissions.Extension)
	if err != nil {
		return nil, err
	}
	if len(companies) == 0 {
		twerr := twirp.NewError(twirp.FailedPrecondition, "Extension does not to belong to an organization")
		twerr = twerr.WithMeta("precondition", "not_in_organization")
		return nil, twerr
	}
	companyID := companies[0].Id

	// Find membership of the assigned biling manager
	membership, err := s.Memberships.GetMembership(ctx, companyID, params.BillingManagerTwitchId)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.NewError(twirp.PermissionDenied, "Billing Manager is not in the same organization as the Extension")
	}
	if err != nil {
		return nil, err // internal error
	}

	// Validate role and ensure can process payments
	switch membership.Role {
	case ownerRole, adminRole:
		err := s.validateCanBeBillingManager(ctx, params.BillingManagerTwitchId)
		if err != nil {
			return nil, err
		}
	case billingManagerRole:
		// already validated: "Billing_Manager" role can only be assigned after payments was properly setup
	default:
		twerr := twirp.NewError(twirp.FailedPrecondition, "billing_manager_twitch_id user does not have an eligible role to be assigned billing manager")
		twerr = twerr.WithMeta("precondition", "invalid_role")
		return nil, twerr
	}

	// Create or update extension-billing-manager relationship
	extensionBillingManager := extensionbillingmanagers.ExtensionBillingManager{
		ExtensionClientID:      params.ExtensionClientId,
		BillingManagerTwitchID: params.BillingManagerTwitchId,
	}
	err = s.Backend.SetExtensionBillingManager(ctx, &extensionBillingManager)
	if err != nil {
		return nil, err
	}

	// Load previous extension Owl client data in case we need to rollback
	prevOwlCli, err := s.Owl.GetClient(ctx, params.ExtensionClientId)
	if err != nil {
		return nil, err // internal; owl.ErrInvalidClientID should not happen because it was already validated
	}

	// Update owner_id in Owl, because that is what the payments system will use for monetization
	err = s.Owl.UpdateClientOwner(ctx, owlcli.UpdateClientRequest{
		ClientID:           params.ExtensionClientId,
		OwnerID:            membership.TwitchID,
		GroupID:            companyID,
		RequestingTwitchID: params.RequestingTwitchId,
	})
	if err != nil {
		// errors owl.ErrInvalidClientID or owl.ErrInvalidUserID should not happen because params were already validated
		return nil, err // internal error
	}

	// DB transaction commit
	if err := s.Backend.Commit(ctx); err != nil {
		s.rollbackOwlClient(ctx, prevOwlCli, params.RequestingTwitchId)
		return nil, err
	}

	s.auditExtensionBillingManagerChange(ctx, ExtensionBillingManagerAction{
		CurrentUserTwitchID: params.RequestingTwitchId,
		EntityTwitchID:      params.BillingManagerTwitchId,
		ActionFormat:        "Edit: Assigned billing manager for extension %s",
		ExtensionClientID:   params.ExtensionClientId,
		CompanyID:           companyID,
	})

	return &rbacrpc.SetExtensionBillingManagerResponse{
		ExtensionBillingManager: extensionBillingManager.ToRPC(),
	}, nil
}

func (s *Server) validateSetExtensionBillingManagerParams(ctx context.Context, params *rbacrpc.SetExtensionBillingManagerRequest) error {
	if auth.IsWhitelistAdmin(ctx) {
		params.RequestingTwitchId = auth.GetTwitchID(ctx)
	}

	return errorutil.ValidateRequiredArgs(errorutil.Args{
		{"billing_manager_twitch_id", params.BillingManagerTwitchId},
		{"extension_client_id", params.ExtensionClientId},
		{"requesting_twitch_id", params.RequestingTwitchId},
	})
}

// rollbackOwlClient can be used in a defer statement to revert an Owl client back to the
// original client values. If there's an error with this, it is sent to Rollbar for manual inspection.
func (s *Server) rollbackOwlClient(ctx context.Context, prevClient *oauth2.Client, requestingTwitchID string) {
	var prevOwnerID string
	if prevClient.OwnerID != nil {
		prevOwnerID = strconv.Itoa(*prevClient.OwnerID)
	}

	var prevGroupID string
	if prevClient.GroupID != nil {
		prevGroupID = *prevClient.GroupID
	}

	req := owlcli.UpdateClientRequest{
		ClientID:           prevClient.ClientID,
		OwnerID:            prevOwnerID,
		GroupID:            prevGroupID,
		RequestingTwitchID: requestingTwitchID,
	}
	err := s.Owl.UpdateClientOwner(ctx, req)
	if err != nil {
		// Do not fail the request, so this helper can be used in a defer statement for Owl rollbacks,
		// but make sure to report the revert failure so we can see it in Slack and fix it manually.
		logx.Error(ctx, errx.Wrap(err, fmt.Sprintf("Failed to revert Owl client to values: %+v", req)))
	}
}

type ExtensionBillingManagerAction struct {
	CurrentUserTwitchID string // twitchID of the main actor of the update
	EntityTwitchID      string // twitchID of the billing manager that is being selected for the extension
	ActionFormat        string // description with role parameterized ("%s")
	ExtensionClientID   string
	CompanyID           string
}

func (s *Server) auditExtensionBillingManagerChange(ctx context.Context, a ExtensionBillingManagerAction) {
	if a.CurrentUserTwitchID == "" {
		a.CurrentUserTwitchID = auth.GetTwitchID(ctx)
	}

	s.ActionHistories.InsertActionHistory(ctx, &actionhistories.ActionHistory{
		UserTwitchID: a.CurrentUserTwitchID,
		Action:       fmt.Sprintf(a.ActionFormat, a.ExtensionClientID),
		EntityType:   "UserRole",
		EntityID:     a.EntityTwitchID,
		CompanyID:    common.NewSQLNullString(a.CompanyID),
	})
}
