package rbacrpcserver

import (
	"context"
	"database/sql"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships/membershipsfakes"
	"code.justin.tv/devrel/devsite-rbac/clients/passport/passportfakes"
	"code.justin.tv/devrel/devsite-rbac/clients/users"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/require"
)

func TestApproveCompanyInvite_Success(t *testing.T) {
	server, rbacBackend := NewTestServer()

	fakeMemberships := &membershipsfakes.FakeMemberships{}
	server.Memberships = fakeMemberships
	usersFake := &users.NoopClient{}
	server.Users = usersFake
	fakePassport := &passportfakes.FakeClient{}
	server.Passport = fakePassport

	ctx := context.Background()
	inviteRequest := acceptInviteRequestTemp()
	invite := companyInviteTemp()

	rbacBackend.SelectCompanyInviteReturns(&invite, nil)
	rbacBackend.AnyCompanyInvitesForUserOnOrganizationReturns(true, nil)
	fakeMemberships.ListMembershipsReturns([]memberships.Membership{{}}, 0, nil)
	fakeMemberships.GetMembershipReturns(memberships.Membership{}, sql.ErrNoRows)
	fakePassport.GetTwoFactorEnabledReturns(true, nil)

	resp, err := server.AcceptCompanyInvite(ctx, &inviteRequest)

	require.NoError(t, err)
	require.NotNil(t, resp)
}

func TestApproveCompanyInvite_AlreadyMember(t *testing.T) {
	server, rbacBackend := NewTestServer()

	fakeMemberships := &membershipsfakes.FakeMemberships{}
	server.Memberships = fakeMemberships
	usersFake := &users.NoopClient{}
	server.Users = usersFake
	fakePassport := &passportfakes.FakeClient{}
	server.Passport = fakePassport

	ctx := context.Background()
	inviteRequest := acceptInviteRequestTemp()
	invite := companyInviteTemp()

	rbacBackend.SelectCompanyInviteReturns(&invite, nil)
	rbacBackend.AnyCompanyInvitesForUserOnOrganizationReturns(false, nil)
	fakeMemberships.ListMembershipsReturns([]memberships.Membership{{}}, 0, nil)
	fakeMemberships.GetMembershipReturns(memberships.Membership{}, nil)
	fakePassport.GetTwoFactorEnabledReturns(true, nil)

	resp, err := server.AcceptCompanyInvite(ctx, &inviteRequest)

	require.Error(t, err)
	require.EqualError(t, err, "twirp error already_exists: Membership already exists. Invite deleted")
	require.Nil(t, resp)
}

func TestValidateAcceptCompanyInviteParams(t *testing.T) {
	type alterParams func() *rbacrpc.AcceptCompanyInviteRequest
	for _, testCase := range []struct {
		name      string
		alterFunc alterParams
		success   bool
	}{
		{
			name: "missing-company-invite-id",
			alterFunc: func() *rbacrpc.AcceptCompanyInviteRequest {
				params := acceptInviteRequestTemp()
				params.CompanyInviteId = ""
				return &params
			},
		},
		{
			name: "missing-first-name",
			alterFunc: func() *rbacrpc.AcceptCompanyInviteRequest {
				params := acceptInviteRequestTemp()
				params.FirstName = ""
				return &params
			},
		},
		{
			name: "missing-last-name",
			alterFunc: func() *rbacrpc.AcceptCompanyInviteRequest {
				params := acceptInviteRequestTemp()
				params.LastName = ""
				return &params
			},
		},
		{
			name: "missing-title",
			alterFunc: func() *rbacrpc.AcceptCompanyInviteRequest {
				params := acceptInviteRequestTemp()
				params.Title = ""
				return &params
			},
		},
		{
			name: "missing-requesting-twitch-id",
			alterFunc: func() *rbacrpc.AcceptCompanyInviteRequest {
				params := acceptInviteRequestTemp()
				params.RequestingTwitchId = ""
				return &params
			},
		},
		{
			name: "valid",
			alterFunc: func() *rbacrpc.AcceptCompanyInviteRequest {
				params := acceptInviteRequestTemp()
				return &params
			},
			success: true,
		},
	} {
		t.Run(testCase.name, func(t *testing.T) {
			params := testCase.alterFunc()
			err := validateAcceptCompanyInviteParams(params)
			if testCase.success {
				require.NoError(t, err)
			} else {
				require.Error(t, err)
			}
		})
	}
}

func TestValidateInvite(t *testing.T) {
	server, rbacBackend := NewTestServer()

	usersFake := &users.NoopClient{}
	server.Users = usersFake
	fakePassport := &passportfakes.FakeClient{}
	server.Passport = fakePassport

	ctx := context.Background()
	invite := companyInviteTemp()

	for _, testCase := range []struct {
		name               string
		companyInviteId    string
		requestingTwitchId string
		action             InviteAction
		inviteError        error
		errorMsg           string
	}{
		{
			name:               "missing-invite",
			companyInviteId:    invite.Id,
			requestingTwitchId: "123",
			action:             InviteAccept,
			inviteError:        sql.ErrNoRows,
			errorMsg:           "twirp error not_found: company invite not found",
		},
		{
			name:               "permission-denied",
			companyInviteId:    invite.Id,
			requestingTwitchId: "1235",
			action:             InviteAccept,
			inviteError:        nil,
			errorMsg:           "twirp error permission_denied: user cannot accepted company invite",
		},
		{
			name:               "valid",
			companyInviteId:    invite.Id,
			requestingTwitchId: "123",
			action:             InviteAccept,
			inviteError:        nil,
			errorMsg:           "",
		},
	} {
		t.Run(testCase.name, func(t *testing.T) {
			rbacBackend.SelectCompanyInviteReturns(&invite, testCase.inviteError)
			_, err := validateInvite(ctx, server, testCase.companyInviteId, testCase.requestingTwitchId, testCase.action)
			if testCase.errorMsg == "" {
				require.NoError(t, err)
			} else {
				require.Error(t, err)
				require.EqualError(t, err, testCase.errorMsg)
			}
		})
	}
}

func companyInviteTemp() rbacrpc.CompanyInvite {
	return rbacrpc.CompanyInvite{
		Id:              common.NewUUID(),
		CompanyId:       common.NewUUID(),
		InviteeTwitchId: "123",
		InviterTwitchId: "123",
		Role:            "Developer",
		CreatedAt:       "2020-04-20 21:56:00",
	}
}

func acceptInviteRequestTemp() rbacrpc.AcceptCompanyInviteRequest {
	return rbacrpc.AcceptCompanyInviteRequest{
		CompanyInviteId:    common.NewUUID(),
		FirstName:          "FirstName",
		LastName:           "LastName",
		Title:              "Title",
		RequestingTwitchId: "123",
	}
}
