package rbacrpcserver

import (
	"context"

	"code.justin.tv/devrel/devsite-rbac/internal/auth"
	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/models/permissions"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"

	"github.com/twitchtv/twirp"
)

func (s *Server) UpdateUserRole(ctx context.Context, params *rbacrpc.UpdateUserRoleRequest) (*rbacrpc.Membership, error) {
	if err := s.validateUpdateUserRoleParams(ctx, params); err != nil {
		return nil, err
	}

	// Ensure permissions
	if !auth.IsWhitelistAdmin(ctx) {
		addUserPermission := permissions.AddUser

		if params.Role == shadowAccountRole {
			// Role Shadow_Account is not allowed to be assigned to an existing user.
			// The role can only be bond with a new user by the RPC CreateShadowAccount.
			return nil, twirp.NewError(twirp.InvalidArgument, "Role Shadow_Account is not allowed to be assigned to an existing user.")
		}

		if params.Role == billingManagerRole {
			addUserPermission = permissions.AddUserBillingManager
		}

		// Get requesting user role in company
		requestingMemb, err := s.Memberships.GetMembership(ctx, params.CompanyId, params.RequestingTwitchId)
		if errorutil.IsErrNoRows(err) {
			return nil, twirp.NewError(twirp.PermissionDenied, "membership for current user (requesting_twitch_id, company_id) not found")
		}
		if err != nil {
			return nil, err
		}

		if !doesRoleHavePermission(requestingMemb.Role, addUserPermission) {
			return nil, twirp.NewError(twirp.PermissionDenied, "cannot update membership role")
		}

		// If the current users role is Manager", respond with  permission denied
		if requestingMemb.Role == managerRole {
			if params.Role == adminRole {
				return nil, twirp.NewError(twirp.PermissionDenied, "cannot update member with role "+params.Role)
			}
		}
	}

	// DB Transaction
	ctx = s.Backend.Begin(ctx)
	defer s.Backend.Rollback(ctx)

	// Load membership to update the role
	memb, err := s.Memberships.GetMembership(ctx, params.CompanyId, params.TwitchId)
	if errorutil.IsErrNoRows(err) {
		return nil, twirp.NotFoundError("membership with company_id and twitch_id not found")
	}
	if err != nil {
		return nil, err
	}

	// If the target user is shadow account, do not allow role update
	if memb.Role == shadowAccountRole {
		return nil, twirp.NewError(twirp.InvalidArgument, "cannot update member to shadow account role, please create a new shadow account instead")
	}

	// If the new role is "Billing_Manager", make sure it has payments enabled
	if params.Role == billingManagerRole {
		if err := s.validateCanBeBillingManager(ctx, params.TwitchId); err != nil {
			return nil, err
		}
	}

	// Owner role can not be updated. Must be changed by staff-admin internally
	if memb.Role == ownerRole {
		return nil, twirp.NewError(twirp.PermissionDenied, "cannot change owner's role")
	}

	// Assigned billing managers can not be updated. Must assign another member before updating the role.
	isAssignedBIllingManager, err := s.Backend.IsAssignedBillingManager(ctx, params.CompanyId, params.TwitchId)
	if err != nil {
		return nil, err
	}
	if isAssignedBIllingManager {
		return nil, twirp.NewError(twirp.FailedPrecondition, "member is an assigned billing manager of an extension")
	}

	// Update membership role
	memb.Role = params.Role
	err = s.Memberships.UpdateMembership(ctx, &memb)
	if err != nil {
		return nil, err
	}

	if err := s.Backend.Commit(ctx); err != nil {
		return nil, err
	}

	s.auditUserRoleChange(ctx, UserRoleAction{
		CurrentUserTwitchID: params.RequestingTwitchId,
		EntityTwitchID:      params.TwitchId,
		ActionFormat:        "Edit: Assigned company role of %s",
		Role:                params.Role,
		CompanyID:           params.CompanyId,
	})
	return memb.ToRPC(), nil
}

func (s *Server) validateUpdateUserRoleParams(ctx context.Context, params *rbacrpc.UpdateUserRoleRequest) error {
	if auth.IsWhitelistAdmin(ctx) {
		params.RequestingTwitchId = auth.GetTwitchID(ctx)
	}

	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"twitch_id", params.TwitchId},
		{"role", params.Role},
		{"company_id", params.CompanyId},
		{"requesting_twitch_id", params.RequestingTwitchId},
	}); err != nil {
		return err
	}

	if err := errorutil.ValidateUUID("company_id", params.CompanyId); err != nil {
		return err
	}

	// Validate role, and make sure the param is properly formatted to simplify equality checks
	formattedRole, err := SanitizeRole("role", params.Role)
	if err != nil {
		return err
	}
	params.Role = formattedRole

	// Cannot change role to Owner
	if params.Role == ownerRole {
		return twirp.NewError(twirp.PermissionDenied, "Cannot update user role to owner")
	}

	return nil
}

// validateCanBeBillingManager checks if the user with that twitchID can be assigned the "Billing_Manager" role,
// that is, if their accounts are ready to receive payments, checking for TIMs and 2FA.
// Returns twirp.FailedPrecondition error if not ready, nil if ready, or another error if there was an internal issue.
func (s *Server) validateCanBeBillingManager(ctx context.Context, twitchID string) error {
	// Requests from the edge (Twilight/GraphQL) send a Twitch-Authorization header with the cartmanToken,
	// our auth middleware puts that token in the context.
	// Requests from Vienna don't have a token, so we need to call Cartman, pretend RBAC is an edge too.
	// The Monepenny service only needs the "cartman::authenticate_first_party" capability, that is used from GQL.
	cartmanToken := auth.GetCartmanToken(ctx)
	if cartmanToken == "" {
		var err error
		oauthToken := auth.GetRawAuthorizationToken(ctx)
		capabilities := "cartman::authenticate_first_party"
		cartmanToken, err = s.Cartman.GetToken(ctx, oauthToken, capabilities, nil)
		if err != nil {
			return err
		}
	}

	// TIMs started and completed?
	timsEnabled, err := s.Moneypenny.HasTIMSEnabled(ctx, twitchID, cartmanToken)
	if errorutil.StatusCode(err) == 403 {
		return twirp.NewError(twirp.PermissionDenied, "Not allowed to see TIMS status")
	}
	if err != nil {
		return err
	}
	if !timsEnabled {
		twerr := twirp.NewError(twirp.FailedPrecondition, "TIMS required for Billing Manager role")
		twerr = twerr.WithMeta("precondition", "tims_required")
		return twerr
	}

	// 2FA: Supposedly TIMS can only be completed with 2FA, but we double check just in case
	twoFactorEnabled, err := s.Passport.GetTwoFactorEnabled(ctx, twitchID)
	if err != nil {
		return err
	}
	if !twoFactorEnabled {
		twerr := twirp.NewError(twirp.FailedPrecondition, "2FA required for Billing Manager role")
		twerr = twerr.WithMeta("precondition", "2fa_required")
		return twerr
	}

	return nil
}
