package rbacrpcserver

import (
	"context"
	"errors"
	"testing"

	"code.justin.tv/devrel/devsite-rbac/clients/cartman"
	"code.justin.tv/devrel/devsite-rbac/rpc/testutil"
	"code.justin.tv/foundation/twitchclient"

	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"

	"code.justin.tv/devrel/devsite-rbac/clients/moneypenny/moneypennyfakes"
	"code.justin.tv/devrel/devsite-rbac/clients/passport/passportfakes"
)

func TestValidateCanBeBillingManager(t *testing.T) {
	m := &moneypennyfakes.FakeClient{}
	p := &passportfakes.FakeClient{}
	c := &cartman.NoopClient{}
	s := &Server{
		Moneypenny: m,
		Passport:   p,
		Cartman:    c,
	}
	ctx := context.Background()
	t.Run("fails", func(t *testing.T) {
		t.Run("when tims is not enabled", func(t *testing.T) {
			m.HasTIMSEnabledReturns(false, nil)
			require.Error(t, s.validateCanBeBillingManager(ctx, "1"))
		})

		t.Run("when tims returns 403 error", func(t *testing.T) {
			m.HasTIMSEnabledReturns(false, &twitchclient.Error{StatusCode: 403})
			err := s.validateCanBeBillingManager(ctx, "1")
			testutil.EqualErrorCode(t, twirp.PermissionDenied, err)
		})

		t.Run("when tims fails", func(t *testing.T) {
			m.HasTIMSEnabledReturns(false, errors.New("asdf"))
			require.Error(t, s.validateCanBeBillingManager(ctx, "1"))
		})

		t.Run("when 2fa is not enabled", func(t *testing.T) {
			m.HasTIMSEnabledReturns(true, nil)
			p.GetTwoFactorEnabledReturns(false, nil)
			require.Error(t, s.validateCanBeBillingManager(ctx, "1"))
		})

		t.Run("when 2fa fails", func(t *testing.T) {
			m.HasTIMSEnabledReturns(true, nil)
			p.GetTwoFactorEnabledReturns(false, errors.New("asdf"))
			require.Error(t, s.validateCanBeBillingManager(ctx, "1"))
		})
	})

	t.Run("succeeds when 2fa and tims are enabled", func(t *testing.T) {
		m.HasTIMSEnabledReturns(true, nil)
		p.GetTwoFactorEnabledReturns(true, nil)
		require.NoError(t, s.validateCanBeBillingManager(ctx, "1"))
	})

}
