package rbacrpcserver

import (
	"context"
	"strings"

	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/models/permissions"

	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/twitchtv/twirp"
)

func (s *Server) ValidateByTwitchID(ctx context.Context, query *rbacrpc.ValidateQuery) (*rbacrpc.ValidateResponse, error) {
	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"user_id", query.UserId}, // this is the TwitchID
		{"resource_type", query.ResourceType},
		{"resource_id", query.ResourceId},
		{"permission", query.Permission},
	}); err != nil {
		return nil, err
	}
	query.ResourceType = strings.ToLower(query.ResourceType) // case insensitive

	var (
		valid bool
		err   error
	)
	switch query.ResourceType {
	case permissions.Company:
		valid, err = s.validateCompanyPermission(ctx, query)
	case permissions.User:
		valid, err = s.validateUserPermission(ctx, query)
	default:
		valid, err = s.validateResourcePermission(ctx, query)
	}
	if err != nil {
		return nil, err
	}
	return &rbacrpc.ValidateResponse{
		Valid: valid,
	}, nil
}

// validates if a user has permission on a company.
func (s *Server) validateCompanyPermission(ctx context.Context, query *rbacrpc.ValidateQuery) (bool, error) {
	var (
		requestingTwitchID = query.UserId
		companyID          = query.ResourceId
		permission         = query.Permission
	)

	// Load membership
	memb, err := s.Memberships.GetMembership(ctx, companyID, requestingTwitchID)
	if errorutil.IsErrNoRows(err) {
		return false, nil // no membership, no access to anything on this company
	}
	if err != nil {
		return false, err
	}

	// Does the role have the permission?
	return doesRoleHavePermission(memb.Role, permission), nil
}

// validates if a user has permission on another user.
func (s *Server) validateUserPermission(ctx context.Context, query *rbacrpc.ValidateQuery) (bool, error) {
	var (
		requestingTwitchID = query.UserId
		twitchID           = query.ResourceId
		permission         = query.Permission
	)

	// Vienna-whitelisted-users can always see payment onboarding status
	if permission == permissions.ViewPaymentOnboarding {
		_, err := s.ViennaUserWhitelist.GetWhitelistedUser(ctx, requestingTwitchID)
		// if err not-found, then is not a vienna whitelisted user, keep going..
		// other errors (internal) are ignored, assuming not a vienna user, to avoid breaking regular access checks.
		if err == nil {
			return true, nil // is a vienna-whitelisted-user
		}
	}

	// Memberships of requesting twitch ID (could be > 1 when multi-org is enabled)
	reqMembs, _, err := s.Memberships.ListMemberships(ctx, memberships.ListMembershipsParams{
		TwitchID: requestingTwitchID,
		Limit:    100,
	})
	if err != nil {
		return false, err
	}
	if len(reqMembs) == 0 {
		return false, nil // no memberships, no access
	}

	for _, reqMemb := range reqMembs {
		// Check if user has the appropriate role in the org for this permission
		if !doesRoleHavePermission(reqMemb.Role, permission) {
			continue
		}

		// ViewPaymentOnboarding permission allows to see anyone, even in other orgs, even outside of RBAC.
		if permission == permissions.ViewPaymentOnboarding {
			return true, nil
		}

		// Check if target member is in the same company
		_, err := s.Memberships.GetMembership(ctx, reqMemb.CompanyID, twitchID)
		if errorutil.IsErrNoRows(err) {
			continue // check next org, requestTwitchID and twitchID could still be together in another org
		}
		if err != nil {
			return false, err // internal error
		}
		return true, nil // requestTwitchID and twitchID are in the same company, all good
	}

	// requestingTwitchID either doesn't have role or twitchID isn't in company
	return false, nil
}

// validates if a user has permission to a resource.
func (s *Server) validateResourcePermission(ctx context.Context, query *rbacrpc.ValidateQuery) (bool, error) {
	var (
		resourceType = query.ResourceType
		resourceID   = query.ResourceId
		twitchID     = query.UserId // requesting twitch id
		permission   = query.Permission
	)

	// load resources
	resources, _, err := s.Backend.FindResources(ctx, "", resourceType, resourceID, 10, 0)
	if err != nil {
		return false, err
	}

	// signify we don't know about the resource
	if len(resources) == 0 {
		return false, twirp.NotFoundError("resource")
	}

	// Memberships of requesting twitch ID (could be > 1 when multi-org is enabled)
	membs, _, err := s.Memberships.ListMemberships(ctx, memberships.ListMembershipsParams{
		TwitchID: twitchID,
		Limit:    100,
	})
	if err != nil {
		return false, err
	}
	if len(membs) == 0 {
		return false, nil // no memberships, no access
	}

	rolesByCompanyID := make(map[string]string)
	for _, memb := range membs {
		rolesByCompanyID[memb.CompanyID] = memb.Role
	}

	for _, resource := range resources {
		role, ok := rolesByCompanyID[resource.CompanyID]
		if !ok {
			// resource may be owned by another company user is a member of
			continue
		}

		ok = doesRoleHavePermission(role, permission)
		if !ok {
			// user can be in another company that has permission
			continue
		}

		return true, nil
	}

	return false, nil
}
