package rbacrpcserver

import (
	"context"
	"database/sql"
	"testing"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/devrel/devsite-rbac/backend/common"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships"
	"code.justin.tv/devrel/devsite-rbac/backend/memberships/membershipsfakes"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	"github.com/stretchr/testify/require"
)

func TestValidateCompanyMembershipByTwitchID_Success_ReturnTrue(t *testing.T) {
	server, _ := NewTestServer()
	ctx := context.Background()
	fakeMemberships := &membershipsfakes.FakeMemberships{}
	server.Memberships = fakeMemberships

	uuid := common.NewUUID()
	request := rbacrpc.ValidateCompanyMembershipByTwitchIDRequest{
		UserId:    "123",
		CompanyId: uuid,
	}

	fakeMemberships.GetMembershipReturns(memberships.Membership{
		CompanyID: uuid,
		Role:      "Developer",
		FirstName: "John",
	}, nil)

	resp, err := server.ValidateCompanyMembershipByTwitchID(ctx, &request)
	require.NoError(t, err)

	require.Equal(t, true, resp.Valid)
}

func TestValidateCompanyMembershipByTwitchID_RequiredArgs(t *testing.T) {
	server, _ := NewTestServer()
	uuid := common.NewUUID()
	ctx := context.Background()

	_, err := server.ValidateCompanyMembershipByTwitchID(ctx, &rbacrpc.ValidateCompanyMembershipByTwitchIDRequest{
		CompanyId: uuid,
	})
	require.EqualError(t, err, "twirp error invalid_argument: user_id is required")

	_, err = server.ValidateCompanyMembershipByTwitchID(ctx, &rbacrpc.ValidateCompanyMembershipByTwitchIDRequest{
		UserId: "123",
	})
	require.EqualError(t, err, "twirp error invalid_argument: company_id is required")

}

func TestValidateCompanyMembershipByTwitchID_WrongOrg_ReturnFalse(t *testing.T) {
	server, _ := NewTestServer()
	ctx := context.Background()
	fakeMemberships := &membershipsfakes.FakeMemberships{}
	server.Memberships = fakeMemberships

	uuidRequested := common.NewUUID()
	uuidReturned := common.NewUUID()
	request := rbacrpc.ValidateCompanyMembershipByTwitchIDRequest{
		UserId:    "123",
		CompanyId: uuidRequested,
	}

	fakeMemberships.GetMembershipReturns(memberships.Membership{
		CompanyID: uuidReturned,
		Role:      "Developer",
		FirstName: "John",
	}, nil)

	resp, err := server.ValidateCompanyMembershipByTwitchID(ctx, &request)
	require.NoError(t, err)

	require.Equal(t, false, resp.Valid)
}

func TestValidateCompanyMembershipByTwitchID_NoOrg_ReturnFalse(t *testing.T) {
	server, _ := NewTestServer()
	ctx := context.Background()
	fakeMemberships := &membershipsfakes.FakeMemberships{}
	server.Memberships = fakeMemberships

	uuid := common.NewUUID()
	request := rbacrpc.ValidateCompanyMembershipByTwitchIDRequest{
		UserId:    "123",
		CompanyId: uuid,
	}

	fakeMemberships.GetMembershipReturns(memberships.Membership{}, errx.Wrap(sql.ErrNoRows, "not found"))

	resp, err := server.ValidateCompanyMembershipByTwitchID(ctx, &request)
	require.NoError(t, err)

	require.Equal(t, false, resp.Valid)
}
