package rbacrpcserver

import (
	"context"
	"strconv"

	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	owl "code.justin.tv/web/owl/client"
	"github.com/twitchtv/twirp"
)

func (s *Server) ValidateExtensionAccess(ctx context.Context, params *rbacrpc.ValidateExtensionAccessRequest) (*rbacrpc.ValidateExtensionAccessResponse, error) {
	// Validate params
	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"user_id", params.UserId},
		{"extension_id", params.ExtensionId},
		{"permission", params.Permission},
	}); err != nil {
		return nil, err
	}
	userIDInt, err := strconv.Atoi(params.UserId)
	if err != nil || userIDInt <= 0 {
		return nil, twirp.InvalidArgumentError("user_id", "expected to be a valid numeric identifier")
	}

	// Check if the extension is in the same org and the permission is correct.
	validateResp, rbacErr := s.ValidateByTwitchID(ctx, &rbacrpc.ValidateQuery{
		UserId:       params.UserId,
		ResourceType: "extension",
		ResourceId:   params.ExtensionId,
		Permission:   params.Permission,
	})
	if rbacErr != nil && errorutil.TwirpCodeFrom(rbacErr) != twirp.NotFound {
		return nil, rbacErr // internal error
	}
	if rbacErr == nil {
		return &rbacrpc.ValidateExtensionAccessResponse{
			Valid:    validateResp.Valid, // permission granted, or not. Maybe the extension is owned by a different org. Maybe the user doesn't have the right privilege.
			OrgOwned: true,
		}, nil
	}

	// rbacErr code is twirp.NotFound, which means RBAC doesn't know about this resource.
	// Check Owl to see if the extension is a personal (org-less) extension instead.

	clientInfo, owlErr := s.Owl.GetClient(ctx, params.ExtensionId)
	if errorutil.Is(owlErr, owl.ErrForbiddenClientID) {
		return nil, twirp.InvalidArgumentError("extension_id", "is an invalid client_id")
	}
	if errorutil.Is(owlErr, owl.ErrInvalidClientID) {
		return nil, twirp.NotFoundError("extension not found")
	}
	if owlErr != nil {
		return nil, twirp.InternalErrorWith(owlErr)
	}

	ownedByCurrentUser := clientInfo.OwnerID != nil && *clientInfo.OwnerID == userIDInt
	return &rbacrpc.ValidateExtensionAccessResponse{
		Valid:    ownedByCurrentUser,
		OrgOwned: false,
	}, nil
}
