package rbacrpcserver

import (
	"context"

	"code.justin.tv/devrel/devsite-rbac/internal/errorutil"
	"code.justin.tv/devrel/devsite-rbac/rpc/rbacrpc"
	clientutils "code.justin.tv/web/users-service/client"
	"github.com/twitchtv/twirp"
)

func (s *Server) IsUserValidForCompanyInvite(ctx context.Context, params *rbacrpc.ValidUserCompanyInviteRequest) (*rbacrpc.ValidUserCompanyInviteResponse, error) {

	response := &rbacrpc.ValidUserCompanyInviteResponse{}

	// verify the user id is real
	userProps, errUser := s.Users.GetUserByID(ctx, params.TwitchId)
	if clientutils.IsUserNotFound(errUser) {
		return nil, twirp.NewError(twirp.NotFound, "user does not exists")
	}
	if errUser != nil {
		return nil, errUser
	}

	// verify the user's email is verified
	if userProps.EmailVerified != nil {
		response.VerifiedEmail = *userProps.EmailVerified
	}

	// verify the user is not already invited to the current company
	anyInvites, err := s.Backend.AnyCompanyInvitesForUserOnOrganization(ctx, params.CompanyId, params.TwitchId)
	response.NoInvites = !anyInvites
	if err != nil {
		return nil, err
	}

	// verify the user is not already invited to the current company
	validMembership, err := s.CheckWithinUserMembershipLimit(ctx, params.TwitchId)
	if err != nil {
		return nil, err
	}
	response.WithinMembershipLimit = validMembership

	// verify the user is not already a member of the company
	_, errMemb := s.Memberships.GetMembership(ctx, params.CompanyId, params.TwitchId)
	response.NotExistingMember = errorutil.IsErrNoRows(errMemb)
	if !response.NotExistingMember && err != nil {
		return nil, err
	}

	// verify user has 2fa enabled
	twoFactorEnabled, err2fa := s.Passport.GetTwoFactorEnabled(ctx, params.TwitchId)
	if err2fa != nil {
		return nil, err2fa
	}
	response.Has_2Fa = twoFactorEnabled

	return response, nil
}

func (s *Server) IsUserQualifiedForBillingManager(ctx context.Context, twitchId *rbacrpc.Id) (*rbacrpc.BillingManagerQualifiedResponse, error) {
	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"twitch id", twitchId.Id},
	}); err != nil {
		return nil, err
	}

	response := &rbacrpc.BillingManagerQualifiedResponse{
		BillingManagerAllowed: false,
	}

	if err := s.validateCanBeBillingManager(ctx, twitchId.Id); err == nil {
		response.BillingManagerAllowed = true
		return response, nil
	}
	return response, nil
}

func (s *Server) IsUserWithinCompanyMembershipLimits(ctx context.Context, twitchId *rbacrpc.Id) (*rbacrpc.IsWithinCompanyMembershipLimitsResponse, error) {
	if err := errorutil.ValidateRequiredArgs(errorutil.Args{
		{"twitch id", twitchId.Id},
	}); err != nil {
		return nil, err
	}

	response := &rbacrpc.IsWithinCompanyMembershipLimitsResponse{
		IsWithinCompanyMembershipLimits: false,
	}

	withinLimits, err := s.CheckWithinUserMembershipLimit(ctx, twitchId.Id)
	if err != nil {
		return nil, err
	}

	response.IsWithinCompanyMembershipLimits = withinLimits

	return response, nil
}
