package testutil

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"
)

func EqualErrorCode(t *testing.T, code twirp.ErrorCode, err error, msgAndArgs ...interface{}) {
	twerr := twirpError(t, err)
	if msgAndArgs == nil {
		msgAndArgs = []interface{}{"Unexpected Twirp error code"}
	}
	require.Equal(t, code, twerr.Code(), msgAndArgs...)
}

func EqualErrorMsg(t *testing.T, msg string, err error, msgAndArgs ...interface{}) {
	twerr := twirpError(t, err)
	if msgAndArgs == nil {
		msgAndArgs = []interface{}{"Unexpected Twirp error msg"}
	}
	require.Equal(t, msg, twerr.Msg(), msgAndArgs...)
}

func EqualErrorMeta(t *testing.T, err error, metaKey, metaValue string, msgAndArgs ...interface{}) {
	twerr := twirpError(t, err)
	if msgAndArgs == nil {
		msg := fmt.Sprintf("Unexpected twerr.Meta(%q) on Twirp error code: %q, msg: %q", metaKey, twerr.Code(), twerr.Msg())
		msgAndArgs = []interface{}{msg}
	}
	require.Equal(t, metaValue, twerr.Meta(metaKey), msgAndArgs...)
}

func twirpError(t *testing.T, err error) twirp.Error {
	require.Error(t, err, "Expected an error to be type-casted to a Twirp error, but it is nil")

	twerr, ok := err.(twirp.Error)
	if !ok {
		// non-twirp errors will be wrapped as internal errors
		// before being returned by the API. Simulate the same here.
		twerr = twirp.InternalErrorWith(err)
	}
	return twerr
}
