package dbx

import (
	"context"
	"database/sql"
	"errors"

	"github.com/Masterminds/squirrel"
	"github.com/jmoiron/sqlx"
)

// ErrManyRows is returned by methods that load or update one record when affecting more than one.
var ErrManyRows = errors.New("dbx: more than one rows affected")

// Values is an alias for syntax sugar, when maps are used instead of structs with db tags.
type Values = map[string]interface{}

// DBX handles rows on a *sqlx.DB connection.
type DBX struct {
	DB *sqlx.DB
}

// DBOrTx returns the DB to execute queries, or a transaction from the ctx if available.
// Transactions are started and added to the context with dbx.MustBegin(ctx, DB).
func (d *DBX) DBOrTx(ctx context.Context) sqlx.Ext {
	if tx := GetActiveTx(ctx); tx != nil {
		return tx
	}
	return d.DB
}

// LoadOne makes a query using sqlx.Get, filling dest with the first result of the query.
// If the context has an active transaction, the query is executed as part of the transaction.
// Returns sql.ErrNoRows if not found.
func (d *DBX) LoadOne(ctx context.Context, dest interface{}, q squirrel.SelectBuilder) error {
	sql, args, err := q.ToSql()
	if err != nil {
		return err
	}
	return sqlx.Get(d.DBOrTx(ctx), dest, sql, args...)
}

// LoadAll makes a query using sqlx.Select, filling dest with all the results of the query.
// If the context has an active transaction, the query is executed as part of the transaction.
// If nothing is found, dest becomes an empty slice (does NOT return an error in this case).
func (d *DBX) LoadAll(ctx context.Context, dest interface{}, q squirrel.SelectBuilder) error {
	sql, args, err := q.ToSql()
	if err != nil {
		return err
	}
	return sqlx.Select(d.DBOrTx(ctx), dest, sql, args...)
}

// InsertOne inserts a new row in the table using sqlx.NamedExec with the given values.
// The values are a struct with `db` tags, or dbx.Values (map[string]interface{}).
// The opts argument is an optional whitelist (dbx.Only) or blacklist (dbx.Exclude) of some values.
// Examples:
//     d.InsertOne(ctx, "heroes", dbx.Values{"id": 1, "name": "Batman"})
//     d.InsertOne(ctx, "heroes", batman) // inserts all fields in batman struct with a `db` tag.
//     d.InsertOne(ctx, "heroes", batman, dbx.Only("id", "name", "created_at")) // only inserts id, name and created_at.
//     d.InsertOne(ctx, "heroes", batman, dbx.Exclude("updated_at")) // all fields exclulding updated_at.
func (d *DBX) InsertOne(ctx context.Context, table string, values interface{}, opts ...FieldsOpt) error {
	fields := FieldsFrom(values).FilterWithOpts(opts...)
	sql := "INSERT INTO " + table +
		" (" + fields.Join(", ") + ")" +
		" VALUES (" + fields.Mapf(":?").Join(", ") + ")"
	return d.NamedExecOne(ctx, sql, values)
}

// UpdateOne updates one row in the table using sqlx.NamedExec.
// The values are a struct with `db` tags, or dbx.Values (map[string]interface{}).
// The opts argument must include dbx.FindBy to specify the primary key (or keys) that uniquely identifies the row to be updated.
// The opts argument can also include an optional whitelist (dbx.Only) or blacklist (dbx.Exclude) of some values.
// Returns sql.ErrNoRows if nothing was updated, or dbx.ErrManyRows if more than one row was updated.
// Examples:
//     d.UpdateOne(ctx, "heroes", batman, dbx.FindBy("id")) // upates all fields, using the "id" field (primary key) to find the record.
//     d.UpdateOne(ctx, "heroes", batman, dbx.FindBy("id"), dbx.Only("name", "updated_at")) // only update name and updated_at.
//     d.UpdateOne(ctx, "heroes", batman, dbx.FindBy("id"), dbx.Exclude("created_at")) // update all, excluding created_at.
//     d.UpdateOne(ctx, "heroes", batman, dbx.FindBy("id", "universe")) // using a composite key ("id = :id AND universe = :universe").
//     d.UpdateOne(ctx, "heroes", dbx.Values{"id": 1, "name": "Batman Forever"}, dbx.FindBy("id")) // using a map instead of a struct
func (d *DBX) UpdateOne(ctx context.Context, table string, values interface{}, opts ...FieldsOpt) error {
	pkFields := FieldsOptsFindBy(opts...)
	if len(pkFields) == 0 {
		panic("dbx.UpdateOne requires a dbx.FindBy primary key, for example: dbx.FindBy(\"id\")")
	}
	setFields := FieldsFrom(values).FilterWithOpts(opts...).Exclude(pkFields...)
	sql := "UPDATE " + table +
		" SET " + setFields.Mapf("? = :?").Join(", ") +
		" WHERE " + pkFields.Mapf("? = :?").Join(" AND ")
	return d.NamedExecOne(ctx, sql, values)
}

// DeleteOne deletes a row in the table using sqlx.NamedExec.
// The values are used as primary key to uniquely identify the row to be deleted.
// The option dbx.FindBy can be used to filter the primary key values.
// Returns sql.ErrNoRows if nothing was deleted, or dbx.ErrManyRows if more than one row was deleted.
// Examples:
//     d.DeleteOne(ctx, "heroes", dbx.Values{"id": 1}) // delete by id.
//     d.DeleteOne(ctx, "heroes", hero, dbx.FindBy("id")) // delete by id, using a struct.
//     d.DeleteOne(ctx, "heroes", hero, dbx.FindBy("id", "universe")) // using a composite key ("id = :id AND universe = :universe").
func (d *DBX) DeleteOne(ctx context.Context, table string, values interface{}, opts ...FieldsOpt) error {
	pkFields := FieldsOptsFindBy(opts...)
	if len(pkFields) == 0 { // if no PK was specified, use all field values as primary key
		pkFields = FieldsFrom(values).FilterWithOpts(opts...)
	}

	sql := "DELETE FROM " + table +
		" WHERE " + pkFields.Mapf("? = :?").Join(" AND ")
	return d.NamedExecOne(ctx, sql, values)
}

// NamedExecOne runs sqlx.NamedExec on the db or current ctx transaction and makes sure that only one row was affected.
// Returns sql.ErrNoRows if no rows were affected.
// Returns an error prefixed with "Affected more than one row" if more than one row was affected.
func (d *DBX) NamedExecOne(ctx context.Context, namedSQL string, values interface{}) error {
	result, err := d.NamedExec(ctx, namedSQL, values)
	if err != nil {
		return err
	}

	// Verify only 1 row was affected
	x, err := result.RowsAffected()
	if err != nil {
		return err
	}
	if x == 0 {
		return sql.ErrNoRows
	}
	if x > 1 {
		return ErrManyRows
	}
	return nil
}

// NamedExec runs sqlx.NamedExec on the db or current ctx transaction.
// For example, if item is type struct{ ID string `db:"id"`, Name string `db:"name"`}, then you could do:
//     d.NamedExec(ctx, "UPDATE items SET name = :name WHERE id = :id", item)
func (d *DBX) NamedExec(ctx context.Context, namedSQL string, values interface{}) (sql.Result, error) {
	return sqlx.NamedExec(d.DBOrTx(ctx), namedSQL, values)
}
