package rollbar

import (
	"net/http"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/stvp/rollbar"
	"golang.org/x/net/context"
)

// ErrorLogger send errors to external tracking services
type ErrorLogger interface {
	Log(context.Context, string, error)
}

type rollbarLogger struct{}
type noopRollbarLogger struct{}

// NewLogger creates a rollbar logger
func NewLogger(token, env string) ErrorLogger {
	if env == "development" {
		return noopRollbarLogger{}
	}
	rollbar.Token = token
	rollbar.Environment = env
	return rollbarLogger{}
}

func (l rollbarLogger) Log(ctx context.Context, severity string, err error) {
	stack := rollbar.BuildStack(0)
	success := make(chan bool, 1)
	errs := hystrix.Go("rollbar_request", func() error {
		if r, ok := ctx.Value("request").(*http.Request); ok {
			rollbar.RequestErrorWithStack(severity, r, err, stack)
		} else {
			rollbar.ErrorWithStack(severity, err, stack)
		}
		success <- true
		return nil
	}, nil)

	select {
	case <-success:
		return
	case <-errs:
		return
	}
}

func (l noopRollbarLogger) LogInit(string, error)              {}
func (l noopRollbarLogger) Log(context.Context, string, error) {}
