#### Passport `errors` Package

This `errors` package replaces and builds upon the golang standard `errors` package by wrapping errors to include client-facing translatable phrases, explicitly treating common errors as pre-defined values for simpler comparison logic, and allowing us to easily extend our wrapped errors to include other features (i.e. error codes).

The goals of this package are to:

1. Reduce error handling logic in controllers to be able to appropriately log errors and easily return translatable error messages to the client.

2. Maintain a single source of error values to allow reuse of common errors in separate packages.

3. Set a foundation for the eventual usage of documented error codes.

### Importing
```
import "code.justin.tv/identity/passport/passport/errors"
```

### General Usage
Wrapped errors contain:

- Logging severity level
- Internal error message
- Translatable phrase

To effectively make use of the additional information in a wrapped error, we could format our controller logic like so:

```
// ValidLogin checks if a login is not an empty string & checks if it already exists.
func ValidLogin(login string) errors.Wrapper {
  if login == "" {
    return errors.InvalidLogin
  }
  exists := CheckIfLoginExists(login)
  if exists {
    return errors.UsernameExists
  }
  return nil
}

func ControllerLogic(ctx context.Context, login string){
  lang := helpers.LangFromContext(ctx)

  err := ValidLogin(login)
  if err != nil{
    logger.Log(err.LogLevel(), err.Error())
    writeToClient(err.Translate(lang))
  }
}
```

The above logic allows the controller to handle errors without having to worry about what the error specifically is.

**As a general rule, all functions that will return errors to a controller should return `errors.Wrapper` instead of `error`**.


### One-off Usage
Ideally, all errors used in Passport would be defined in the errors package; however, it is common to come across situations where a specific error will only occur once. For these specific cases, the errors package replicates the functionality of the standard golang `errors` package's `errors.New` as well as the functionality of `fmt.Errorf`.
```
// Create a new one-off wrapped error
err := errors.New("this is a newly wrapped error with default log level and default external facing message.")
logger.Log(err.LogLevel(), err.Error())
```

```
// Wrap an error with formatting
err := doSomethingWrong()
wrappedErr := errors.Errorf("this is a newly wrapped error with formatting %s", err)
```

```
// Wrap an error without formatting
err := doSomethingWrong()
wrappedErr := errors.Wrap(err)
```
