package errors

import (
	"code.justin.tv/identity/passport/logger"
	i18n "code.justin.tv/web/go-i18n"
)

var (
	// AmazonUnexpected represents a general error raised during Amazon authentication.
	AmazonUnexpected = &WrappedError{
		InternalError: "unexpected error with amazon",
		ExternalMessage: i18n.T(
			"Oops! You authenticated via Amazon correctly, but we encountered an error. Please try again.",
			"Error text displayed when an unexpected error occurs when attempting to auth via amazon.",
		),
		Severity: defaultLogLevel,
		Code:     6001,
	}

	// AmazonAuth represents an error when we fail to validate Amazon token.
	AmazonAuth = &WrappedError{
		InternalError: "failed to validate amazon token",
		ExternalMessage: i18n.T(
			"Oops! We've encountered an unexpected error authenticating you with Amazon. Please try again.",
			"Error text displayed when we run into an error when talking with Amazon.",
		),
		Severity: defaultLogLevel,
		Code:     6002,
	}

	// AmazonUserNotFound represents a failure to return a user_id given an amazon identifier.
	AmazonUserNotFound = &WrappedError{
		InternalError: "no twitch user found for amazon id",
		ExternalMessage: i18n.T(
			"Oops! We were unable to verify your identity with Amazon. Please try again.",
			"Error text displayed when the user enters bad information when logging via Amazon",
		),
		Severity: defaultLogLevel,
		Code:     6003,
	}

	// AmazonConnectFailure represents an error raised when we are unable to create a link
	// between amazon id and twitch user.
	AmazonConnectFailure = &WrappedError{
		InternalError: "unable to connect twitch user to amazon id",
		ExternalMessage: i18n.T(
			"Oops! You authenticated via Amazon correctly, but we encountered an error. Please try again.",
			"Error text displayed when an unexpected error occurs when attempting to auth via amazon.",
		),
		Severity: logger.Warn,
		Code:     6004,
	}

	// AmazonTooManyLinks represents an error when a user attempts to connect too many Twitch accounts
	// to a specific amazon id.
	AmazonTooManyLinks = &WrappedError{
		InternalError: "amazon id linked to too many twitch users",
		ExternalMessage: i18n.T(
			"This amazon account is already linked to a user.",
			"Error text displayed when a user attempts to link a twitch account with amazon.",
		),
		Severity: defaultLogLevel,
		Code:     6005,
	}

	// AmazonMultipleLinks represents an error when an Amazon account maps to multiple Twitch accounts
	AmazonMultipleLinks = &WrappedError{
		InternalError: "multiple twitch accounts linked to amazon id",
		ExternalMessage: i18n.T(
			"There are multiple Twitch accounts linked to this Amazon ID",
			"Error text displayed when a user attempts to login to Twitch with an Amazon token.",
		),
		Severity: defaultLogLevel,
		Code:     6006,
	}

	// AmazonMissingFields represents an error when there are missing required parameters
	AmazonMissingFields = &WrappedError{
		InternalError: "missing required parameters for operation",
		ExternalMessage: i18n.T(
			"Missing parameters",
			"Error text displayed when a user attempts to log in or sign up to Twitch with an Amazon token.",
		),
		Severity: defaultLogLevel,
		Code:     6007,
	}

	// AmazonAccountAlreadyLinked represents an error when a user attempts to link an account that is already linked
	AmazonAccountAlreadyLinked = &WrappedError{
		InternalError: "account already linked",
		ExternalMessage: i18n.T(
			"Amazon account has already been linked to this Twitch ID",
			"Error text displayed when a user attempts to log in or sign up to Twitch with an Amazon token.",
		),
		Severity: defaultLogLevel,
		Code:     6008,
	}
)
