package errors

import (
	"code.justin.tv/identity/passport/logger"
	i18n "code.justin.tv/web/go-i18n"
)

// Registered OAuth 2.0 error codes derived from IANA Error Registry
// http://www.iana.org/assignments/oauth-parameters/oauth-parameters.xhtml
const (
	InvalidRequestCode   = "invalid_request"
	InvalidTokenCode     = "invalid_token"
	RedirectMismatchCode = "redirect_mismatch"
	ServerErrorCode      = "server_error"
	ServerErrorDesc      = "Server encountered error when processing request"
)

var (
	// BadToken indicates that either the header has a non matching algorithm,
	// the signature is bad, or the encoding is bad.
	// TOOD: update external message as it's outdated
	BadToken = &WrappedError{
		InternalError: "invalid token",
		ExternalMessage: i18n.T(
			"Oops! Your credentials look correct, but we encountered an error logging you in. Please try again.",
			"Error text given when user submits bad information to the /token endpoint.",
		),
		Severity: defaultLogLevel,
		Code:     4000,
	}

	// BadClaim indicates that the claims made in the token body are invalid.
	// TOOD: update external message as it's outdated
	BadClaim = &WrappedError{
		InternalError: "invalid claim",
		ExternalMessage: i18n.T(
			"Oops! Your credentials look correct, but we encountered an error logging you in. Please try again.",
			"Error text given when user submits bad information to the /token endpoint.",
		),
		Severity: defaultLogLevel,
		Code:     4001,
	}

	// MissingUsername -> request sent with empty username.
	MissingUsername = &WrappedError{
		InternalError: "missing username",
		ExternalMessage: i18n.T(
			"Please enter a username.",
			"Error text displayed for a password reset error",
		),
		Severity: defaultLogLevel,
		Code:     4002,
	}

	// MissingEmail -> request sent with empty email.
	MissingEmail = &WrappedError{
		InternalError: "missing email",
		ExternalMessage: i18n.T(
			"Please enter an email.",
			"Error text displayed for a password reset error",
		),
		Severity: defaultLogLevel,
		Code:     4003,
	}

	// InvalidResetToken indicates the reset token is incorrectly associated with a reset request.
	InvalidResetToken = &WrappedError{
		InternalError: "invalid reset token",
		ExternalMessage: i18n.T(
			"Invalid reset token. Please access this page directly from the password reset email.",
			"Password reset error",
		),
		Severity: defaultLogLevel,
		Code:     4004,
	}

	InvalidCurrentPassword = &WrappedError{
		InternalError: "invalid current password",
		ExternalMessage: i18n.T(
			"Your current password was incorrect",
			"Password change error",
		),
		Severity: defaultLogLevel,
		Code:     4005,
	}

	// SessionsUnexpected is an error raised when failing to communicate with Sessions service.
	SessionsUnexpected = &WrappedError{
		InternalError:   "unexpected error with Sessions service",
		ExternalMessage: DefaultExternalMessage,
		Severity:        logger.Warn,
		Code:            4006,
	}

	// TOOD: can this be tossed out, it's is only used once
	LoginChangeOnCooldown = &WrappedError{
		InternalError: "login change on cooldown",
		ExternalMessage: i18n.T(
			"You are unable to update your username at this time.",
			"Login change error",
		),
		Severity: defaultLogLevel,
		Code:     4007,
	}

	// TOOD: can this be tossed out, it's is only used once
	LoginChangeEmailFailure = &WrappedError{
		InternalError: "failed to send login change confirmation email",
		ExternalMessage: i18n.T(
			"Sorry, we couldn't send a confirmation email.",
			"Login change error",
		),
		Severity: defaultLogLevel,
		Code:     4008,
	}

	// NoEmail represents a failure to retrieve an email address for the Twitch username.
	NoEmail = &WrappedError{
		InternalError: "there is no email associated with that user",
		ExternalMessage: i18n.T(
			"Sorry, we don't have an email on file for that user account.",
			"Password reset error",
		),
		Severity: defaultLogLevel,
		Code:     4009,
	}

	// PasswordReuseDisallowed represents a user trying to reuse the same password when not allowed
	PasswordReuseDisallowed = &WrappedError{
		InternalError: "password reused during forced reset",
		ExternalMessage: i18n.T(
			"Your new password must be different from your old password.",
			"Password reset error",
		),
		Severity: defaultLogLevel,
		Code:     4010,
	}

	// ResetEmailFailure represents a generic error raised while sending password reset email.
	ResetEmailFailure = &WrappedError{
		InternalError: "failed to send password reset email",
		ExternalMessage: i18n.T(
			"Sorry, we couldn't send a password reset email. Please try again in a minute.",
			"Password reset error",
		),
		Severity: defaultLogLevel,
		Code:     4011,
	}

	VerifiedEmailRequired = &WrappedError{
		InternalError: "verified email required",
		ExternalMessage: i18n.T(
			"A verified email address is required. Please verify your email address and try again.",
			"Login change error",
		),
		Severity: defaultLogLevel,
		Code:     4012,
	}

	// ForgotUsernameThrottled is an error returned when an email or IP is throttled
	// on the forgot username flow.
	ForgotUsernameThrottled = &WrappedError{
		InternalError: "invalid auth token",
		ExternalMessage: i18n.T(
			"You have requested too many username reminders recently. Please try again later.",
			"Throttling message displayed to a user when they ask for too many username reminder emails in a short period of time",
		),
		Severity: logger.Info,
		Code:     4013,
	}

	PasswordChangeThrottled = &WrappedError{
		InternalError: "too many password change attempts",
		ExternalMessage: i18n.T(
			"You've gotten your current password wrong too many times recently. Please try again in a few minutes.",
			"Password reset error",
		),
		Severity: defaultLogLevel,
		Code:     4014,
	}

	// ResetLoginThrottled represents an error due to too many password reset attempts in a short period of time.
	ResetLoginThrottled = &WrappedError{
		InternalError: "too many password reset attempts for this login",
		ExternalMessage: i18n.T(
			`You've requested too many password resets today. Please try again in 24hrs or <a href=\"https://help.twitch.tv/customer/portal/emails/new?ticket[custom_category]=logging%20in%20to%20my%20account\">Contact Us</a>.`,
			"Password reset error",
		),
		Severity: defaultLogLevel,
		Code:     4015,
	}

	// ResetIPThrottled represents an error due to too many password reset attempts in a short period of time.
	ResetIPThrottled = &WrappedError{
		InternalError: "too many password reset attempts for this ip",
		ExternalMessage: i18n.T(
			`You've requested too many password resets recently! Please try again in five minutes or <a href=\"https://help.twitch.tv/customer/portal/emails/new?ticket[custom_category]=logging%20in%20to%20my%20account\">Contact Us</a>.`,
			"Password reset error",
		),
		Severity: defaultLogLevel,
		Code:     4016,
	}
	// ResetAttempts represents a failure to reset attempts.
	ResetAttempts = &WrappedError{
		InternalError:   "unable to reset attempts",
		ExternalMessage: DefaultExternalMessage,
		Severity:        defaultLogLevel,
		Code:            4017,
	}

	// ResetTimeout represents a failure to reset timeout.
	ResetTimeout = &WrappedError{
		InternalError:   "unable to reset timeout",
		ExternalMessage: DefaultExternalMessage,
		Severity:        defaultLogLevel,
		Code:            4018,
	}

	// InvalidLoginEmailCombo represents an attempt to password reset with an email that's not associated with the username.
	InvalidLoginEmailCombo = &WrappedError{
		InternalError: "incorrect email address for username",
		ExternalMessage: i18n.T(
			"Sorry, we couldn't find a user account for that username and email combination.",
			"Password reset error",
		),
		Severity: defaultLogLevel,
		Code:     4019,
	}

	LoginChangeThrottled = &WrappedError{
		InternalError: "too many login change attempts",
		ExternalMessage: i18n.T(
			"You've gotten your current password wrong too many times recently. Please try again in a few minutes.",
			"Login change error",
		),
		Severity: defaultLogLevel,
		Code:     4020,
	}

	// LogoutCSRF is returned when the api_token cookie value did not match the
	// api_token form value. Likely due to a CSRF attempt.
	LogoutCSRF = &WrappedError{
		InternalError:   "cookie does not match form value in POST request",
		ExternalMessage: DefaultExternalMessage,
		Severity:        logger.Trace,
		Code:            4021,
	}

	// SessionsClientError is an error raised when there is bad input into Sessions Service.
	// Possible due to a bad JSON body or an Invalid JSON token
	SessionsClientError = &WrappedError{
		InternalError:   "unexpected client error with Sessions service",
		ExternalMessage: DefaultExternalMessage,
		Severity:        logger.Warn,
		Code:            4022,
	}

	// InvalidOAuthToken is an error raised when Owl fails to validate given OAuth token
	InvalidOAuthToken = &WrappedError{
		InternalError:   "invalid oauth token for spare key exchange",
		ExternalMessage: DefaultExternalMessage,
		Severity:        logger.Info,
		Code:            4023,
	}

	// InvalidSpareKeyClient is an error raised when a valid oauth token for an invalid oauth client
	// is attempted to be exchanged for a spare key cookie.
	InvalidSpareKeyClient = &WrappedError{
		InternalError:   "invalid client for spare key exchange",
		ExternalMessage: DefaultExternalMessage,
		Severity:        logger.Info,
		Code:            4024,
	}

	// SpareKeyUnexpected is an error raised when failing to set spare key cookie
	SpareKeyUnexpected = &WrappedError{
		InternalError:   "failed to set spare key cookie",
		ExternalMessage: DefaultExternalMessage,
		Severity:        logger.Warn,
		Code:            4025,
	}

	// NotEligibleForAccountRecovery is an error raised when account is not eligible for phone number recovery
	NotEligibleForAccountRecovery = &WrappedError{
		InternalError:   "account not eligible for phone number recovery",
		ExternalMessage: DefaultExternalMessage,
		Severity:        logger.Info,
		Code:            4026,
	}
	// NotEligibleForAccountRecovery is an error raised when account is not eligible for phone number recovery and also lacks an email
	NotEligibleForAccountRecoveryAndNoEmail = &WrappedError{
		InternalError:   "account not eligible for phone number recovery and also lacks an email",
		ExternalMessage: DefaultExternalMessage,
		Severity:        logger.Info,
		Code:            4027,
	}
)
