package errors

import (
	"code.justin.tv/identity/passport/logger"
	i18n "code.justin.tv/web/go-i18n"
)

var (
	LoginUnexpected = &WrappedError{
		InternalError: "unexpected error with login",
		ExternalMessage: i18n.T(
			"Oops! We encountered an unexpected error logging you in. Please try again.",
			"Error text displayed when an unexpected problem happens during login.",
		),
		Severity: defaultLogLevel,
		Code:     3000,
	}

	IncorrectCredentials = &WrappedError{
		InternalError: "user credentials incorrect",
		ExternalMessage: i18n.T(
			"Incorrect username or password.",
			"Error text displayed when a user inputs a bad username or password into the login form.",
		),
		Severity: logger.Trace,
		Code:     3001,
	}

	MissingLogin = &WrappedError{
		InternalError: "username missing",
		ExternalMessage: i18n.T(
			"Please enter your username and password.",
			"Error text displayed when a user submits a login form with missing information",
		),
		Severity: logger.Trace,
		Code:     3002,
	}

	MissingPassword = &WrappedError{
		InternalError: "password missing",
		ExternalMessage: i18n.T(
			"Please enter your username and password.",
			"Error text displayed when a user submits a login form with missing information",
		),
		Severity: logger.Trace,
		Code:     3003,
	}

	MultipleLinkedAccounts = &WrappedError{
		InternalError:   "multiple accounts linked to email",
		ExternalMessage: IncorrectCredentials.ExternalMessage,
		Severity:        logger.Trace,
		Code:            3004,
	}

	NoLinkedAccount = &WrappedError{
		InternalError:   "no account linked to email or phone number",
		ExternalMessage: IncorrectCredentials.ExternalMessage,
		Severity:        logger.Trace,
		Code:            3005,
	}

	NeedPasswordReset = &WrappedError{
		InternalError:   "user needs password reset",
		ExternalMessage: DefaultExternalMessage,
		Code:            3006,
	}

	UserDeleted = &WrappedError{
		InternalError: "user has been deleted",
		ExternalMessage: i18n.T(
			"Incorrect username or password.",
			"Error text displayed when a user inputs a bad username or password into the login form.",
		),
		Severity: logger.Trace,
		Code:     3007,
	}

	SuspendedUser = &WrappedError{
		InternalError: "suspended user",
		Severity:      defaultLogLevel,
		Code:          3008,
	}

	// DeviceCookieThrottled is returned when a user is throttled for too many
	// failed login attempts. This can be returned whether or not a valid
	// device cookie is present.
	DeviceCookieThrottled = &WrappedError{
		InternalError: "device cookie throttle reached",
		ExternalMessage: i18n.T(
			"You have attempted to log in to this account too many times. Please try again later.",
			"Error text displayed when a a user is throttled for failing to authenticate too many times",
		),
		Severity: defaultLogLevel,
		Code:     3009,
	}

	CodeUnmarshal = &WrappedError{
		InternalError:   "failed to unmarshal 2fa code",
		ExternalMessage: DefaultExternalMessage,
		Severity:        logger.Info,
		Code:            3010,
	}

	MissingAuthyToken = &WrappedError{
		InternalError: "missing authy token",
		ExternalMessage: i18n.T(
			"Missing authentication token. Please try again.",
			"Error text displayed when a user submits a two factor form when missing a two factor token.",
		),
		Severity: logger.Trace,
		Code:     3011,
	}

	InvalidAuthyToken = &WrappedError{
		InternalError: "invalid authy token",
		ExternalMessage: i18n.T(
			"Invalid authentication token. Please try again.",
			"Error text displayed when the token submitted by the user is incorrect.",
		),
		Severity: logger.Info,
		Code:     3012,
	}

	VerifyAuthyToken = &WrappedError{
		InternalError: "failed to verify authy token",
		ExternalMessage: i18n.T(
			"Could not verify your authentication token. Please try again.",
			"Error text displayed when we are un able to verify the two factor token with authy.",
		),
		Severity: logger.Warn,
		Code:     3013,
	}

	SMSUnexpected = &WrappedError{
		InternalError: "failed to deliver sms",
		ExternalMessage: i18n.T(
			"Error encountered when sending SMS. Please try again.",
			"Text displayed when we are unable to send an sms",
		),
		Severity: logger.Warn,
		Code:     3014,
	}

	SMSThrottled = &WrappedError{
		InternalError: "sms code requests throttled",
		ExternalMessage: i18n.T(
			"You have requested too many SMS messages. Try again later.",
			"Error text displayed when the user has asked for too many SMS verification codes.",
		),
		Severity: logger.Info,
		Code:     3015,
	}

	UnsupportedAuthyMethod = &WrappedError{
		InternalError: "unsupported authy 2fa requested",
		ExternalMessage: i18n.T(
			"Requested 2FA type is not supported",
			"Error text displayed when the user has asks for an invalid 2FA type.",
		),
		Severity: logger.Info,
		Code:     3016,
	}
)
