package errors

import i18n "code.justin.tv/web/go-i18n"

var (
	// InvalidAuthorization encapsulates the lack of an authorization or an expired
	// authorization between a given Client and end user
	InvalidAuthorization = &WrappedError{
		InternalError: "authorization does not exist or is expired",
		Severity:      defaultLogLevel,
		Code:          9900,
	}

	// InvalidAuthCode represents an error when a requesting client presents an expired
	// or invalid authorization code in exchange for a set of OIDC / OAuth Tokens
	InvalidAuthorizationCode = &WrappedError{
		InternalError: "invalid authorization code",
		Severity:      defaultLogLevel,
		Code:          9901,
	}

	// InvalidClientCredentials is an OAuth Client error where
	// a requesting client presents an invalid pair of id and secret
	InvalidClientCredentials = &WrappedError{
		InternalError: "invalid client credentials",
		Severity:      defaultLogLevel,
		Code:          9902,
	}

	// InvalidResponseType is an OAuth error where a requesting client
	// doesn't supply a recognized OAuth 2.0 response type cvalue
	InvalidOAuthResponseType = &WrappedError{
		InternalError: "OAuth response type is not supported",
		Severity:      defaultLogLevel,
		Code:          9903,
	}

	// MissingRedirectURI is an OAuth error where a requesting client
	// doesn't supply an expected redirect_uri
	MissingRedirectURI = &WrappedError{
		InternalError: "missing redirect_uri",
		Severity:      defaultLogLevel,
		ExternalMessage: i18n.T(
			"Missing redirect_uri",
			"Error text displayed when a client does not send a redirect_uri when one is expected",
		),
		Code: 9904,
	}

	// UnsupportedGrantType is an OAuth error where a requesting client
	// doesn't supply a grant_type that we support
	UnsupportedGrantType = &WrappedError{
		InternalError: "unsupported grant_type",
		Severity:      defaultLogLevel,
		Code:          9905,
	}

	//UnauthorizedClient is an error thrown when an unrecognized 3rd part attempts to make an authorization request
	UnauthorizedClient = &WrappedError{
		InternalError:   "invalid client attempting to make an authorization request",
		ExternalMessage: DefaultExternalMessage,
		Severity:        defaultLogLevel,
		Code:            9906,
	}

	// LoginThrottled is returned when a user has failed to authenticate
	// too many times in the OAuth password credentials grant flow.
	LoginThrottled = &WrappedError{
		InternalError: "too many failed login attempts",
		Severity:      defaultLogLevel,
		Code:          9907,
	}

	// MissingAuthyTokenPWGrant is an OAuth error for the password credentials grant flow
	// when a 2fa user does not provide their 2fa token with their password
	MissingAuthyTokenPWGrant = &WrappedError{
		InternalError: "please add your two-factor authentication token to the end of your password",
		Severity:      defaultLogLevel,
		Code:          9908,
	}

	// ErrNoAccessToken is an OAuth error when the access token is missing
	ErrNoAccessToken = &WrappedError{
		InternalError: "no access token present in request",
		Severity:      defaultLogLevel,
		Code:          9909,
	}
)
