package errors

import (
	"code.justin.tv/identity/passport/logger"
	i18n "code.justin.tv/web/go-i18n"
)

var (
	// SignupUnexpected represents a general error raised while attempting to sign up new user.
	SignupUnexpected = &WrappedError{
		InternalError: "unexpected issue with signup",
		ExternalMessage: i18n.T(
			"Oops! We were unable to create your account. Please try again.",
			"error text when an unknown error happens when signing a user up",
		),
		Severity: logger.Warn,
		Code:     2000,
	}

	// BlacklistedIP is an error representing a failure to create an account due to prior blacklisting
	// of the user's IP address.
	BlacklistedIP = &WrappedError{
		InternalError:   "blacklisted IP",
		ExternalMessage: DefaultExternalMessage,
		Severity:        defaultLogLevel,
		Code:            2001,
	}

	// InvalidEmailDomain respresents an errors raised when the user attempts to signup up
	// with a blacklisted email domain (i.e. mailnesia)
	InvalidEmailDomain = &WrappedError{
		InternalError: "signup attempted with blacklisted email domain",
		ExternalMessage: i18n.T(
			"Please enter a different email address.",
			"error text when a user inputs an invalid email address",
		),
		Severity: defaultLogLevel,
		Code:     2002,
	}

	// InvalidBirthday represents a signup attempt with an invalid birthday.
	InvalidBirthday = &WrappedError{
		InternalError: "invalid birthday",
		ExternalMessage: i18n.T(
			"This birthday does not seem to be valid.",
			"error text when a user attempts to sign up with a birthday within the last 4 years",
		),
		Severity: defaultLogLevel,
		Code:     2003,
	}

	// InvalidUsername is an error representing an invalid username in signup/login forms.
	InvalidUsername = &WrappedError{
		InternalError: "invalid username",
		ExternalMessage: i18n.T(
			"Invalid username.",
			"Error text displayed when a user inputs an invalid username.",
		),
		Severity: defaultLogLevel,
		Code:     2004,
	}

	// InvalidPassword -> password not between 8 and 71 characters, or fails zxcvbn strength check.
	InvalidPassword = &WrappedError{
		InternalError: "invalid password",
		ExternalMessage: i18n.T(
			"Password too weak. You must use a stronger password to continue.",
			"Invalid password error",
		),
		Severity: defaultLogLevel,
		Code:     2005,
	}

	MissingSignupFields = &WrappedError{
		InternalError: "missing fields",
		ExternalMessage: i18n.T(
			"Please fill out the signup form completely.",
			"Error text displayed for missing fields on signup form",
		),
		Severity: defaultLogLevel,
		Code:     2006,
	}

	// TooManyUsersAssociatedWithEmail is an error raised when an account creation fails due to that
	// email address being associated with 10 or more emails.
	TooManyUsersAssociatedWithEmail = &WrappedError{
		InternalError: "too many users associated with this email",
		ExternalMessage: i18n.T(
			"Too many accounts have been associated with this email address",
			"Error text displayed when a user tries to sign up with an overused email address.",
		),
		Severity: defaultLogLevel,
		Code:     2007,
	}

	// UsernameExists represents a general error raised while attempting to sign up new user.
	UsernameExists = &WrappedError{
		InternalError: "username already exists",
		ExternalMessage: i18n.T(
			"This username already exists.",
			"error text when user attempts to sign up with an existing username",
		),
		Severity: logger.Trace,
		Code:     2008,
	}

	// UsernameTooShort represents an error raised when attempting to sign up with a short username.
	UsernameTooShort = &WrappedError{
		InternalError: "username too short",
		ExternalMessage: i18n.T(
			"Username is too short.",
			"error text when user attempts to sign up a short username",
		),
		Severity: logger.Trace,
		Code:     2009,
	}

	// UsernameTooLong represents an error raised when attempting to sign up with a long username.
	UsernameTooLong = &WrappedError{
		InternalError: "username too long",
		ExternalMessage: i18n.T(
			"Username is too long.",
			"error text when user attempts to sign up with a long username",
		),
		Severity: logger.Trace,
		Code:     2010,
	}

	// Under13 represents an attempt by a user that has the under13 cookie set on their browser.
	Under13 = &WrappedError{
		InternalError: "user is under 13",
		ExternalMessage: i18n.T(
			"You are not eligible to create an account at this time.",
			"error text when an underage user attempts to sign up.",
		),
		Severity: defaultLogLevel,
		Code:     2011,
	}

	// UserExistsFailed represents a failure to check if username is available.
	UserExistsFailed = &WrappedError{
		InternalError:   "unable to check if user exists",
		ExternalMessage: DefaultExternalMessage,
		Severity:        defaultLogLevel,
		Code:            2012,
	}

	// SignupEmailThrottled represents an error raised when the user attempts to sign up too many
	// times with the same email address within a configured window.
	SignupEmailThrottled = &WrappedError{
		InternalError: "email associated with signup throttled",
		ExternalMessage: i18n.T(
			"This email has been associated with too many account creations recently.",
			"error text when user has associated this email with too many accounts",
		),
		Severity: defaultLogLevel,
		Code:     2013,
	}

	// SignupIPThrottled represents an error raised when the user attempts to sign up too many
	// times with the same email address within a configured window.
	SignupIPThrottled = &WrappedError{
		InternalError: "ip associated with signup throttled",
		ExternalMessage: i18n.T(
			"You are trying to sign up for accounts too fast.",
			"error text when user has associated this ip with too many accounts",
		),
		Severity: defaultLogLevel,
		Code:     2014,
	}

	// EmailDisabledForReuse represents an error raised when the user attempts to create additional
	// accounts with the same email address and additional account creation has not been enabled.
	EmailDisabledForReuse = &WrappedError{
		InternalError: "email disabled for reuse",
		ExternalMessage: i18n.T(
			"Unable to create account. Please use a different email address to continue",
			"error text when user has not enabled email for reuse",
		),
		Severity: defaultLogLevel,
		Code:     2015,
	}

	// PhoneNumberInUse represents an error raised when the user attempts to create additional
	// accounts with the same phone number.
	PhoneNumberInUse = &WrappedError{
		InternalError: "phone number already associated with another account",
		ExternalMessage: i18n.T(
			"Unable to create account. This phone number is already associated with an account",
			"error text when user is trying to register with an phone number that is already in the system",
		),
		Severity: defaultLogLevel,
		Code:     2016,
	}

	// PhoneNumberIsntMobile represents an error raised when the user attempts to create an
	// account with a phone number that is not a mobile phone number.
	PhoneNumberIsntMobile = &WrappedError{
		InternalError: "phone number is not a mobile phone number",
		ExternalMessage: i18n.T(
			"Unable to create account. This phone number is must be a mobile (non-voip) phone number",
			"error text when user is trying to register with an phone number that is not mobile",
		),
		Severity: defaultLogLevel,
		Code:     2017,
	}

	// PhoneNumberUnrecognized represents an error raised when the user attempts to create an
	// account with a phone number that is not of a valid format (as determined by twilio).
	PhoneNumberUnrecognized = &WrappedError{
		InternalError: "phone number is not a valid phone number",
		ExternalMessage: i18n.T(
			"Unable to create account. This phone number does not seem to be a phone number",
			"error text when user is trying to register with an invalidly formatted phone number",
		),
		Severity: defaultLogLevel,
		Code:     2018,
	}
)
