/*
Package errors allows us to predefine reusable errors with additional details for logging
and client-facing messages & wrap existing errors with the same functionality.
*/
package errors

import (
	"fmt"

	"code.justin.tv/identity/passport/logger"
	i18n "code.justin.tv/web/go-i18n"
)

// Default values for WrappedError fields.
var (
	DefaultExternalMessage = i18n.T(
		"Oops! We encountered an unexpected error. Please try again.",
		"Error text displayed when an un expected problem happens in passport.",
	)
	DefaultExternalString = DefaultExternalMessage.Phrase
	defaultLogLevel       = logger.Info
	defaultErrorCode      = 0
)

// Wrapper interface represents an error value that can provide an
// error code and user-readable string.
type Wrapper interface {
	Error() string
	Translate(string) string
	ErrorCode() int
	LogLevel() logger.Severity
	OverrideExternalMessage(*i18n.String)
	OverrideInternalMessage(string)
	OverrideCode(int)
}

// WrappedError is the struct that will implement Wrapper and consist of fields
// holding log level, error code, and user-facing messages.
type WrappedError struct {
	InternalError   string
	Code            int
	ExternalMessage *i18n.String
	Severity        logger.Severity
}

// RailsWrappedError is a workaround for already translated strings that we get
// from Rails and implements Wrapper.
type RailsWrappedError struct {
	*WrappedError
	ExternalMessage string
}

// Implements "error" interface by calling Error() on the wrapped error.
func (e *WrappedError) Error() string {
	return e.InternalError
}

// Translate implements "Wrapper" interface and returns the translated client-facing message.
func (e *WrappedError) Translate(lang string) string {
	return e.ExternalMessage.Translate(lang)
}

// ErrorCode implements "Wrapper" interface and returns the error code.
func (e *WrappedError) ErrorCode() int {
	return e.Code
}

// LogLevel implements "Wrapper" interface and returns the error code.
func (e *WrappedError) LogLevel() logger.Severity {
	return e.Severity
}

// OverrideExternalMessage implements "Wrapper" interface and allows us to set ExternalMessage.
func (e *WrappedError) OverrideExternalMessage(m *i18n.String) {
	e.ExternalMessage = m
}

// OverrideInternalMessage implements "Wrapper" interface and allows us to set InternalMessage.
func (e *WrappedError) OverrideInternalMessage(m string) {
	e.InternalError = m
}

// OverrideCode implements "Wrapper" interface and allows us to set the code.
func (e *WrappedError) OverrideCode(code int) {
	e.Code = code
}

// Translate implements the "Wrapper" interface and returns an already translated string.
func (e *RailsWrappedError) Translate(lang string) string {
	return e.ExternalMessage
}

// New builds a WrappedError with default fields.
func New(errorString string) Wrapper {
	return &WrappedError{
		InternalError:   errorString,
		ExternalMessage: DefaultExternalMessage,
		Severity:        defaultLogLevel,
		Code:            defaultErrorCode,
	}
}

// Wrap encapsulates an error within an errors.Wrapper with default values.
func Wrap(internalError error) Wrapper {
	if internalError == nil {
		return nil
	}

	// If the error has already been wrapped, we won't do anything to it.
	wrapped, ok := internalError.(*WrappedError)
	if ok {
		return wrapped
	}

	railsWrapped, ok := internalError.(*RailsWrappedError)
	if ok {
		return railsWrapped
	}

	return &WrappedError{
		InternalError:   internalError.Error(),
		ExternalMessage: DefaultExternalMessage,
		Severity:        defaultLogLevel,
		Code:            defaultErrorCode,
	}

}

// Errorf builds an error similarly to fmt.Errorf by formatting strings
// according to the format specifier
func Errorf(format string, a ...interface{}) Wrapper {
	return New(fmt.Sprintf(format, a...))
}
