package i18n

import (
	"errors"
	"fmt"
	"log"
	"math"

	"crypto/md5"

	"github.com/go-playground/locales"
	"github.com/go-playground/locales/ar_SA"
	"github.com/go-playground/locales/bg_BG"
	"github.com/go-playground/locales/cs_CZ"
	"github.com/go-playground/locales/da_DK"
	"github.com/go-playground/locales/de_DE"
	"github.com/go-playground/locales/el_GR"
	"github.com/go-playground/locales/es_419"
	"github.com/go-playground/locales/es_ES"
	"github.com/go-playground/locales/fi_FI"
	"github.com/go-playground/locales/fr_FR"
	"github.com/go-playground/locales/hu_HU"
	"github.com/go-playground/locales/it_IT"
	"github.com/go-playground/locales/ja_JP"
	"github.com/go-playground/locales/ko_KR"
	"github.com/go-playground/locales/nb_NO"
	"github.com/go-playground/locales/nl_NL"
	"github.com/go-playground/locales/pl_PL"
	"github.com/go-playground/locales/pt_BR"
	"github.com/go-playground/locales/pt_PT"
	"github.com/go-playground/locales/ro_RO"
	"github.com/go-playground/locales/ru_RU"
	"github.com/go-playground/locales/sk_SK"
	"github.com/go-playground/locales/sv_SE"
	"github.com/go-playground/locales/th_TH"
	"github.com/go-playground/locales/tr_TR"
	"github.com/go-playground/locales/vi_VN"
	"github.com/go-playground/locales/zh_Hans_CN"
	"github.com/go-playground/locales/zh_Hant_TW"
	"github.com/gotnospirit/messageformat"

	"github.com/go-playground/locales/en"
	"io"
	"strings"
	"sync"
)

const defaultLocale = "en"

var (
	// twitch locale => cldr rule set
	ICURulesByTwitchLocale = map[string]locales.Translator{
		"en":    en.New(),
		"ar":    ar_SA.New(),
		"bg":    bg_BG.New(),
		"cs":    cs_CZ.New(),
		"da":    da_DK.New(),
		"de":    de_DE.New(),
		"el":    el_GR.New(),
		"es":    es_ES.New(),
		"es-mx": es_419.New(),
		"fi":    fi_FI.New(),
		"fr":    fr_FR.New(),
		"hu":    hu_HU.New(),
		"it":    it_IT.New(),
		"ja":    ja_JP.New(),
		"ko":    ko_KR.New(),
		"nl":    nl_NL.New(),
		"no":    nb_NO.New(),
		"pl":    pl_PL.New(),
		"pt-br": pt_BR.New(),
		"pt":    pt_PT.New(),
		"ro":    ro_RO.New(),
		"ru":    ru_RU.New(),
		"sk":    sk_SK.New(),
		"sv":    sv_SE.New(),
		"th":    th_TH.New(),
		"tr":    tr_TR.New(),
		"vi":    vi_VN.New(),
		"zh-cn": zh_Hans_CN.New(),
		"zh-tw": zh_Hant_TW.New(),
	}
)

// Making this a different type so only this type will be applied to the icu translation methods
type ICUString String

type ICUTranslator interface {
	// If the translation is found, return: ("translation", true)
	// If translation is not found, return: ("", false)
	Translate(s *ICUString, locale string) (string, bool)
}

type ICUArgs map[string]interface{}

type I18nICU interface {
	CLDRTranslators() map[string]locales.Translator

	// Returns bool to indicate whether the translator for the requested locale was returned
	CLDRTranslatorForLocale(locale string) (locales.Translator, bool)
	TranslationWithArgs(s *ICUString, locale string, icuArgs ICUArgs) (string, error)
}

type i18nICUImpl struct {
	defaultParser  *messageformat.Parser
	icuTranslator  ICUTranslator
	formatterCache sync.Map
	logger         *log.Logger
}

func (i *i18nICUImpl) CLDRTranslators() map[string]locales.Translator {
	return ICURulesByTwitchLocale
}

func (i *i18nICUImpl) CLDRTranslatorForLocale(locale string) (locales.Translator, bool) {
	translator, ok := ICURulesByTwitchLocale[locale]
	if ok {
		return translator, ok
	}

	return ICURulesByTwitchLocale[defaultLocale], false
}

func (i *i18nICUImpl) TranslationWithArgs(s *ICUString, locale string, icuArgs ICUArgs) (string, error) {
	mf, err := i.formatter(s, locale)
	if err != nil {
		return "", err
	}

	return mf.FormatMap(icuArgs)
}

func (i *i18nICUImpl) formatter(s *ICUString, locale string) (*messageformat.MessageFormat, error) {
	locale = i.toValidLocale(locale)
	translation, ok := i.icuTranslator.Translate(s, locale)
	if ok {
		mf, err := i.formatterForLocale(s, translation, locale)
		if err != nil {
			i.logger.Printf("warning, could not get formatter for locale: %v. %v", locale, err)
		} else {
			return mf, nil
		}
	}

	return i.formatterForLocale(s, s.Phrase, defaultLocale)
}

func (i *i18nICUImpl) formatterForLocale(s *ICUString, translation string, locale string) (*messageformat.MessageFormat, error) {
	key := createKey(s, locale)
	// If in cache, use it
	if cacheValue, loaded := i.formatterCache.Load(key); loaded {
		mf, ok := cacheValue.(*messageformat.MessageFormat)
		if ok {
			return mf, nil
		}
		i.logger.Printf("could not cast cache value to message format for key: %v", key)
	}

	// Otherwise create it
	mf, err := i.defaultParser.Parse(translation)
	if err != nil {
		return nil, err
	}

	err = mf.SetPluralFunction(cardinalPluralRuleFn(locale, i.logger))
	if err != nil {
		return nil, err
	}

	// And add to cache
	i.formatterCache.Store(key, mf)

	return mf, nil
}

// Returns 'en' if the locale isn't valid
func (i *i18nICUImpl) toValidLocale(locale string) string {
	if _, ok := ICURulesByTwitchLocale[locale]; !ok {
		return defaultLocale
	}
	return locale
}

func NewI18nICU(t ICUTranslator, logger *log.Logger) (I18nICU, error) {
	defaultParser, err := messageformat.New()
	if err != nil {
		return nil, err

	}

	return &i18nICUImpl{
		defaultParser: defaultParser,
		icuTranslator: t,
		logger:        logger,
	}, nil
}

var _ I18nICU = (*i18nICUImpl)(nil)

func createKey(s *ICUString, locale string) string {
	return fmt.Sprintf("[%s] %s", locale, s.Hash())
}

func toFloat64(val interface{}) (float64, error) {
	switch i := val.(type) {
	case float64:
		return i, nil
	case float32:
		return float64(i), nil
	case int:
		return float64(i), nil
	case int8:
		return float64(i), nil
	case int16:
		return float64(i), nil
	case int32:
		return float64(i), nil
	case int64:
		return float64(i), nil
	default:
		return math.NaN(), errors.New("value is of incompatible type")
	}
}

// Returns which ICU rule should be used for a specific number (zero/one/two/other/few/many/etc)
func cardinalPluralRuleFn(locale string, logger *log.Logger) func(interface{}, bool) string {
	translator := ICURulesByTwitchLocale[locale]

	return func(val interface{}, ordinal bool) string {
		num, err := toFloat64(val)
		if err != nil {
			logger.Println(err.Error())
			return "other"
		}

		return strings.ToLower(translator.CardinalPluralRule(num, 0).String())
	}
}

func ICUT(phrase string, context string) *ICUString {
	return &ICUString{
		Phrase:  phrase,
		Context: context,
	}
}

func (s *ICUString) TranslateICU(i18nICU I18nICU, locale string, icuArgs ICUArgs) (string, error) {
	return i18nICU.TranslationWithArgs(s, locale, icuArgs)
}

func (s *ICUString) TranslateICUSafe(i18nICU I18nICU, locale string, icuArgs ICUArgs) string {
	str, err := s.TranslateICU(i18nICU, locale, icuArgs)
	if err != nil {
		return s.Phrase
	}

	return str
}

// Hash returns a unique hash of an i18n.String
func (s *ICUString) Hash() StringHash {
	if s.hash != "" {
		return s.hash
	}
	h := md5.New()
	if _, err := io.WriteString(h, s.Phrase); err != nil {
		log.Printf("warning: failed to write phrase to md5: %v", err)
	}
	if _, err := io.WriteString(h, s.Context); err != nil {
		log.Printf("warning: failed to write context to md5: %v", err)
	}
	s.hash = StringHash(fmt.Sprintf("%x", h.Sum(nil)))
	return s.hash
}
