package smartling

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"mime/multipart"
	"net/http"
	"net/url"
	"sync"
	"time"
)

// V2 API: https://api-reference.smartling.com/

const (
	defaultSmartlingHost = "https://api.smartling.com/"
)

type smartlingFileType string

// Supported Smartling FileTypes
// android, ios, gettext, html, javaProperties, yaml, xliff, xml, json, docx, pptx, xlsx, idml, qt, resx, plaintext, cvs, stringsdict
const (
	AndroidFileType smartlingFileType = "android"
	IOSFileType     smartlingFileType = "ios"
	JSONFileType    smartlingFileType = "json"
)

type ClientAPI interface {
	GetFile(input *GetFileInput) (*GetFileOutput, error)
	UploadFile(input *UploadFileInput) (*UploadFileOutput, error)
	ListFiles(input *ListFilesInput) (*ListFilesOutput, error)
	GetProjectDetails(input *GetProjectDetailsInput) (*GetProjectDetailsOutput, error)
	Authenticate(input *AuthenticateInput) (*AuthenticateOutput, error)
	RefreshAuthentication(input *RefreshAuthenticationInput) (*RefreshAuthenticationOutput, error)
}

var _ ClientAPI = (*Client)(nil)

// Client is an implementation for the smartling V2 API
type Client struct {
	host       *url.URL
	httpClient *http.Client

	lock       sync.Mutex
	authData   *authData
	userID     string
	userSecret string
}

// Config contains options for the smartling client
type Config struct {
	Host       string
	HTTPClient *http.Client
	UserID     string
	UserSecret string
}

// NewClient returns a new Client instance
func NewClient(config *Config) (*Client, error) {
	if config == nil {
		config = &Config{}
	}

	if config.UserID == "" {
		return nil, errors.New("UserID cannot be blank")
	} else if config.UserSecret == "" {
		return nil, errors.New("UserSecret cannot be blank")
	}

	if config.Host == "" {
		config.Host = defaultSmartlingHost
	}

	if config.HTTPClient == nil {
		config.HTTPClient = http.DefaultClient
	}

	u, err := url.Parse(config.Host)
	if err != nil {
		return nil, err
	}

	return &Client{
		host:       u,
		httpClient: config.HTTPClient,
		userID:     config.UserID,
		userSecret: config.UserSecret,
	}, nil
}

type responseWrapper struct {
	Response Response `json:"response"`
}

// Response is a struct for the standard response format
type Response struct {
	Code   string          `json:"code"`
	Errors []ResponseError `json:"errors"`
	Data   json.RawMessage `json:"data"`

	// V1 deprecated
	Messages []string `json:"messages"`
}

// ResponseError is a struct for the standard error format
type ResponseError struct {
	Key     string            `json:"key"`
	Message string            `json:"message"`
	Details map[string]string `json:"details"`
}

// IsError returns whether the request succeeded
func (r *Response) IsError() bool {
	return r.Code != "SUCCESS"
}

// IsValidationError returns whether the error is a validation error
func IsValidationError(err error) bool {
	if resp, ok := err.(*Response); ok {
		return resp.Code == "VALIDATION_ERROR"
	}

	return false
}

// Error returns an error string. This satisfies the error interface
func (r *Response) Error() string {
	return fmt.Sprintf("smartling error: %s: %#v", r.Code, r.Errors)
}

// GetFileInput is the input struct for the download file API
// https://api-reference.smartling.com/#operation/downloadTranslatedFileSingleLocale
type GetFileInput struct {
	ProjectID              string
	Lang                   string
	FileURI                string
	RetrievalType          string
	IncludeOriginalStrings bool
}

// GetFileOutput is the output struct for the download file API
type GetFileOutput struct {
	Contents []byte
}

// GetFile calls the download file API
func (c *Client) GetFile(input *GetFileInput) (*GetFileOutput, error) {
	if input == nil {
		return nil, errors.New("input cannot be nil")
	} else if input.ProjectID == "" {
		return nil, errors.New("ProjectID cannot be blank")
	} else if input.Lang == "" {
		return nil, errors.New("Lang cannot be blank")
	} else if input.FileURI == "" {
		return nil, errors.New("FileURI cannot be blank")
	}

	req, err := c.newAuthRequest("GET", fmt.Sprintf("/files-api/v2/projects/%s/locales/%s/file", input.ProjectID, input.Lang), nil)
	if err != nil {
		return nil, err
	}

	values := req.URL.Query()
	values.Add("fileUri", input.FileURI)

	if input.RetrievalType != "" {
		values.Add("retrievalType", input.RetrievalType)
	}

	if input.IncludeOriginalStrings {
		values.Add("includeOriginalStrings", "true")
	} else {
		values.Add("includeOriginalStrings", "false")
	}
	req.URL.RawQuery = values.Encode()

	resp, err := c.httpClient.Do(req)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		serr := unmarshalSmartlingResponse(resp, nil)
		if serr != nil {
			return nil, serr
		}

		return nil, fmt.Errorf("http error: %s", resp.Status)
	}

	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	out, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	return &GetFileOutput{
		Contents: out,
	}, nil
}

// UploadFileInput is the input struct for the upload file API
// https://api-reference.smartling.com/#operation/uploadSourceFile
type UploadFileInput struct {
	ProjectID    string
	FileURI      string
	Authorize    bool
	FileType     smartlingFileType
	FileContents io.Reader
	CallbackURL  string
}

// UploadFileOutput is the output struct for the upload file API
type UploadFileOutput struct {
	Overwritten bool `json:"overWritten"`
	StringCount int  `json:"stringCount"`
	WordCount   int  `json:"wordCount"`
}

// UploadFile calls the upload file API
func (c *Client) UploadFile(input *UploadFileInput) (*UploadFileOutput, error) {
	if input == nil {
		return nil, errors.New("input cannot be blank")
	} else if input.ProjectID == "" {
		return nil, errors.New("ProjectID cannot be blank")
	} else if input.FileURI == "" {
		return nil, errors.New("FileURI cannot be blank")
	} else if input.FileType == "" {
		return nil, errors.New("FileType cannot be blank")
	} else if input.FileContents == nil {
		return nil, errors.New("FileContents cannot be blank")
	}

	var body bytes.Buffer
	w := multipart.NewWriter(&body)

	fileWriter, err := w.CreateFormFile("file", input.FileURI)
	if err != nil {
		return nil, err
	}

	_, err = io.Copy(fileWriter, input.FileContents)
	if err != nil {
		return nil, err
	}

	err = w.WriteField("fileUri", input.FileURI)
	if err != nil {
		return nil, err
	}

	if input.Authorize {
		err = w.WriteField("authorize", "true")
	} else {
		err = w.WriteField("authorize", "false")
	}

	if err != nil {
		return nil, err
	}

	err = w.WriteField("fileType", string(input.FileType))
	if err != nil {
		return nil, err
	}

	if input.CallbackURL != "" {
		err = w.WriteField("callbackUrl", input.CallbackURL)
		if err != nil {
			return nil, err
		}
	}

	err = w.Close()
	if err != nil {
		return nil, err
	}

	req, err := c.newAuthRequest("POST", fmt.Sprintf("/files-api/v2/projects/%s/file", input.ProjectID), &body)
	if err != nil {
		return nil, err
	}

	req.Header.Add("Content-Type", w.FormDataContentType())

	var out UploadFileOutput
	err = c.doSmartlingRequest(req, &out)
	if err != nil {
		return nil, err
	}

	return &out, nil
}

// ListFileInput is the input struct for the list file API
// https://api-reference.smartling.com/#operation/downloadTranslatedFileSingleLocale
type ListFilesInput struct {
	ProjectID string
	URIMask   string
}

// ListFileOutput is the output struct for the list file API
type ListFilesOutput struct {
	TotalCount int                  `json:"totalCount"`
	Items      []ListFileOutputItem `json:"items"`
}

// ListFileOutputItem contains data about each project file
type ListFileOutputItem struct {
	FileURI         string    `json:"fileUri"`
	LastUploaded    time.Time `json:"lastUploaded"`
	FileType        string    `json:"fileType"`
	HasInstructions bool      `json:"hasInstructions"`
}

// ListFiles calls the list file API
func (c *Client) ListFiles(input *ListFilesInput) (*ListFilesOutput, error) {
	if input == nil {
		return nil, errors.New("input cannot be nil")
	} else if input.ProjectID == "" {
		return nil, errors.New("ProjectID cannot be blank")
	}

	req, err := c.newAuthRequest("GET", fmt.Sprintf("/files-api/v2/projects/%s/files/list", input.ProjectID), nil)
	if err != nil {
		return nil, err
	}

	values := req.URL.Query()
	if input.URIMask != "" {
		values.Add("uriMask", input.URIMask)
	}
	req.URL.RawQuery = values.Encode()

	var out ListFilesOutput
	err = c.doSmartlingRequest(req, &out)
	if err != nil {
		return nil, err
	}

	return &out, nil
}

// GetProjectDetailsInput is the input struct for the get project details API
// https://api-reference.smartling.com/#operation/downloadTranslatedFileSingleLocale
type GetProjectDetailsInput struct {
	ProjectID string
}

// GetProjectDetailsOutput is the output struct for the get project details API
type GetProjectDetailsOutput struct {
	ProjectID               string         `json:"projectId"`
	ProjectName             string         `json:"projectName"`
	AccountUID              string         `json:"accountUid"`
	SourceLocaleID          string         `json:"sourceLocaleId"`
	SourceLocaleDescription string         `json:"sourceLocaleDescription"`
	Archived                bool           `json:"archived"`
	TargetLocales           []TargetLocale `json:"targetLocales"`
}

// TargetLocale contains locale data
type TargetLocale struct {
	LocaleID    string `json:"localeId"`
	Description string `json:"description"`
}

// GetProjectDetails calls the get project details API
func (c *Client) GetProjectDetails(input *GetProjectDetailsInput) (*GetProjectDetailsOutput, error) {
	if input == nil {
		return nil, errors.New("input cannot be nil")
	} else if input.ProjectID == "" {
		return nil, errors.New("ProjectID cannot be blank")
	}

	req, err := c.newAuthRequest("GET", fmt.Sprintf("/projects-api/v2/projects/%s", input.ProjectID), nil)
	if err != nil {
		return nil, err
	}

	var out GetProjectDetailsOutput
	err = c.doSmartlingRequest(req, &out)
	if err != nil {
		return nil, err
	}

	return &out, nil
}

type authData struct {
	AccessToken           string
	AccessTokenExpiresAt  time.Time
	RefreshToken          string
	RefreshTokenExpiresAt time.Time
}

func (a *authData) isAccessTokenExpired() bool {
	return time.Now().After(a.AccessTokenExpiresAt)
}

func (a *authData) isRefreshTokenExpired() bool {
	return time.Now().After(a.RefreshTokenExpiresAt)
}

// getAccessToken is a thread-safe internal function to get the OAuth token
func (c *Client) getAccessToken() (string, error) {
	c.lock.Lock()
	defer c.lock.Unlock()

	if c.authData == nil {
		err := c.authenticate()
		if err != nil {
			return "", err
		}

		return c.authData.AccessToken, nil
	}

	if c.authData.isAccessTokenExpired() {
		if c.authData.isRefreshTokenExpired() {
			err := c.authenticate()
			if err != nil {
				return "", err
			}
		} else {
			err := c.refreshAuthentication()
			if err != nil {
				return "", err
			}
		}
	}

	return c.authData.AccessToken, nil
}

// authenticate calls the authenticate API and sets internal auth data state
func (c *Client) authenticate() error {
	out, err := c.Authenticate(&AuthenticateInput{
		UserID:     c.userID,
		UserSecret: c.userSecret,
	})

	if err != nil {
		return err
	}

	c.authData = &authData{
		AccessToken:           out.AccessToken,
		AccessTokenExpiresAt:  time.Now().Add(time.Second * time.Duration(out.ExpiresIn)),
		RefreshToken:          out.RefreshToken,
		RefreshTokenExpiresAt: time.Now().Add(time.Second * time.Duration(out.RefreshExpiresIn)),
	}

	return nil
}

// refreshAuthentication calls the refresh authentication API and sets internal auth data state
func (c *Client) refreshAuthentication() error {
	out, err := c.RefreshAuthentication(&RefreshAuthenticationInput{
		RefreshToken: c.authData.RefreshToken,
	})

	if err != nil {
		return err
	}

	c.authData = &authData{
		AccessToken:           out.AccessToken,
		AccessTokenExpiresAt:  time.Now().Add(time.Second * time.Duration(out.ExpiresIn)),
		RefreshToken:          out.RefreshToken,
		RefreshTokenExpiresAt: time.Now().Add(time.Second * time.Duration(out.RefreshExpiresIn)),
	}

	return nil
}

// AuthenticateInput is the input struct for the authenticate API
// https://api-reference.smartling.com/#operation/authenticate
type AuthenticateInput struct {
	UserID     string `json:"userIdentifier"`
	UserSecret string `json:"userSecret"`
}

// AuthenticateOutput is the output struct for the authenticate API
type AuthenticateOutput struct {
	AccessToken      string `json:"accessToken"`
	ExpiresIn        int    `json:"expiresIn"`
	RefreshExpiresIn int    `json:"refreshExpiresIn"`
	RefreshToken     string `json:"refreshToken"`
	TokenType        string `json:"tokenType"`
	SessionState     string `json:"sessionState"`
}

// Authenticate calls the authenticate API
func (c *Client) Authenticate(input *AuthenticateInput) (*AuthenticateOutput, error) {
	if input == nil {
		return nil, errors.New("input cannot be nil")
	} else if input.UserID == "" {
		return nil, errors.New("UserID cannot be blank")
	} else if input.UserSecret == "" {
		return nil, errors.New("UserSecret cannot be blank")
	}

	req, err := c.newJSONRequest("POST", "/auth-api/v2/authenticate", input)

	if err != nil {
		return nil, err
	}

	var out AuthenticateOutput
	err = c.doSmartlingRequest(req, &out)
	if err != nil {
		return nil, err
	}

	return &out, nil
}

// RefreshAuthenticationInput is the input struct for the refresh authentication API
// https://api-reference.smartling.com/#operation/refreshAccessToken
type RefreshAuthenticationInput struct {
	RefreshToken string `json:"refreshToken"`
}

// RefreshAuthenticationOutput is the output struct for the refresh authentication API
type RefreshAuthenticationOutput struct {
	AccessToken      string `json:"accessToken"`
	ExpiresIn        int    `json:"expiresIn"`
	RefreshExpiresIn int    `json:"refreshExpiresIn"`
	RefreshToken     string `json:"refreshToken"`
	TokenType        string `json:"tokenType"`
	SessionState     string `json:"sessionState"`
}

// RefreshAuthentication calls the refresh authentication API
func (c *Client) RefreshAuthentication(input *RefreshAuthenticationInput) (*RefreshAuthenticationOutput, error) {
	if input == nil {
		return nil, errors.New("input cannot be nil")
	} else if input.RefreshToken == "" {
		return nil, errors.New("RefreshToken cannot be nil")
	}

	req, err := c.newJSONRequest("POST", "/auth-api/v2/authenticate/refresh", input)

	if err != nil {
		return nil, err
	}

	var out RefreshAuthenticationOutput
	err = c.doSmartlingRequest(req, &out)
	if err != nil {
		return nil, err
	}

	return &out, nil
}

// doSmartlingRequest executes the HTTP request and unmarshals the response
func (c *Client) doSmartlingRequest(req *http.Request, data interface{}) error {
	resp, err := c.httpClient.Do(req)
	if err != nil {
		return nil
	}

	return unmarshalSmartlingResponse(resp, data)
}

func (c *Client) newJSONRequest(method string, path string, data interface{}) (*http.Request, error) {
	input, err := json.Marshal(data)
	if err != nil {
		return nil, err
	}

	req, err := c.newRequest(method, path, bytes.NewBuffer(input))
	if err != nil {
		return nil, err
	}

	req.Header.Add("Content-Type", "application/json")

	return req, nil
}

func (c *Client) newAuthRequest(method string, path string, body io.Reader) (*http.Request, error) {
	req, err := c.newRequest(method, path, body)
	if err != nil {
		return nil, err
	}

	accessToken, err := c.getAccessToken()
	if err != nil {
		return nil, fmt.Errorf("error getting access token: %v", err)
	}

	req.Header.Add("Authorization", "Bearer "+accessToken)

	return req, nil
}

func (c *Client) newRequest(method string, path string, body io.Reader) (*http.Request, error) {
	u, err := url.Parse(path)
	if err != nil {
		return nil, err
	}

	return http.NewRequest(method, c.host.ResolveReference(u).String(), body)
}

func unmarshalSmartlingResponse(resp *http.Response, data interface{}) error {
	var wrapper responseWrapper
	var err error

	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	err = json.NewDecoder(resp.Body).Decode(&wrapper)
	if err != nil {
		return fmt.Errorf("unable to read response body: %v", err)
	}

	if wrapper.Response.IsError() {
		return &wrapper.Response
	}

	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("http error: %s", resp.Status)
	}

	if data == nil {
		return nil
	}

	return json.Unmarshal(wrapper.Response.Data, data)
}

// DEPRECATED V1 Client
type ClientV1 struct {
	APIKey string
	Host   string
}

func (c *ClientV1) GetFile(input *GetFileInput) (*GetFileOutput, error) {
	uri, err := c.resolveURI("/v1/file/get")
	if err != nil {
		return nil, err
	}

	resp, err := http.PostForm(uri,
		url.Values{
			"apiKey": []string{
				c.APIKey,
			},
			"projectId": []string{
				input.ProjectID,
			},
			"fileUri": []string{
				input.FileURI,
			},
			"locale": []string{
				input.Lang,
			},
		},
	)

	if err != nil {
		return nil, err
	}

	defer func() {
		if cerr := resp.Body.Close(); cerr != nil && err == nil {
			err = cerr
		}
	}()

	if resp.StatusCode != http.StatusOK {
		serr := unmarshalSmartlingResponse(resp, nil)
		if serr != nil {
			return nil, serr
		}

		return nil, fmt.Errorf("http error: %s", resp.Status)
	}

	out, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	return &GetFileOutput{
		Contents: out,
	}, nil
}

func (c *ClientV1) UploadFile(input *UploadFileInput) (*UploadFileOutput, error) {
	uri, err := c.resolveURI("/v1/file/upload")
	if err != nil {
		return nil, err
	}

	var body bytes.Buffer

	w := multipart.NewWriter(&body)

	fileWriter, err := w.CreateFormFile("file", input.FileURI)
	if err != nil {
		return nil, err
	}
	_, err = io.Copy(fileWriter, input.FileContents)
	if err != nil {
		return nil, err
	}

	for key, value := range map[string]string{
		"fileUri":   input.FileURI,
		"projectId": input.ProjectID,
		"apiKey":    c.APIKey,
		"approved":  "true",
		"fileType":  (string)(input.FileType),
	} {
		err = w.WriteField(key, value)
		if err != nil {
			return nil, err
		}
	}

	err = w.Close()
	if err != nil {
		return nil, err
	}

	reqUpload, err := http.NewRequest("POST", uri, &body)
	if err != nil {
		return nil, err
	}

	reqUpload.Header.Add("Content-Type", w.FormDataContentType())

	resp, err := http.DefaultClient.Do(reqUpload)
	if err != nil {
		return nil, err
	}

	var out UploadFileOutput
	err = unmarshalSmartlingResponse(resp, &out)
	if err != nil {
		return nil, err
	}

	return &out, nil
}

func (c *ClientV1) resolveURI(path string) (string, error) {
	hostname := c.Host
	if hostname == "" {
		hostname = defaultSmartlingHost
	}

	host, err := url.Parse(hostname)
	if err != nil {
		return "", err
	}

	u, err := url.Parse(path)
	if err != nil {
		return "", err
	}

	return host.ResolveReference(u).String(), nil
}
