package models

import (
	"net/url"
	"strings"
	"time"

	"golang.org/x/net/publicsuffix"
)

type ChannelSocialMediaResultsRepresentation struct {
	Results []ChannelSocialMediasRepresentation `json:"results"`
	Error   string                              `json:"error"`
}

type ChannelSocialMediasRepresentation struct {
	ID        string    `json:"id"`
	URL       string    `json:"url"`
	Name      *string   `json:"name"`
	Title     string    `json:"title"`
	UserID    string    `json:"user_id"`
	CreatedAt time.Time `json:"created_at"`
	UpdatedAt time.Time `json:"updated_at"`
}

func getSocialMediasNameFromURL(u string) *string {
	url, err := url.Parse(u)
	if err != nil {
		return nil
	}

	ps, _ := publicsuffix.PublicSuffix(url.Host)
	if len(ps) == 0 {
		return nil
	}

	prefix := strings.TrimSuffix(strings.TrimSuffix(url.Host, ps), ".")
	if len(prefix) == 0 {
		return nil
	}

	parts := strings.Split(prefix, ".")
	if len(parts) <= 0 {
		return nil
	}

	domain := parts[len(parts)-1]
	if len(domain) <= 0 {
		return nil
	}

	return &domain
}

func NewChannelSocialMediaResultsRepresentation(cs []ChannelSocialMediaProperties, err error) ChannelSocialMediaResultsRepresentation {
	if err != nil {
		return ChannelSocialMediaResultsRepresentation{Error: err.Error()}
	} else if cs != nil {
		return ChannelSocialMediaResultsRepresentation{Results: NewChannelSocialMediasRepresentations(cs)}
	} else {
		return ChannelSocialMediaResultsRepresentation{}
	}
}

func NewChannelSocialMediasRepresentations(cs []ChannelSocialMediaProperties) []ChannelSocialMediasRepresentation {
	r := []ChannelSocialMediasRepresentation{}

	for _, c := range cs {
		r = append(r, NewChannelSocialMediasRepresentation(&c))
	}

	return r
}

func NewChannelSocialMediasRepresentation(c *ChannelSocialMediaProperties) ChannelSocialMediasRepresentation {
	r := ChannelSocialMediasRepresentation{
		ID:        c.ID,
		URL:       c.URL,
		Title:     c.Title,
		UserID:    c.UserID,
		CreatedAt: c.CreatedAt,
		UpdatedAt: c.UpdatedAt,
	}

	if c.URL != "" {
		r.Name = getSocialMediasNameFromURL(c.URL)
	}

	return r
}
