import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as cloudfront from '@aws-cdk/aws-cloudfront';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecs from '@aws-cdk/aws-ecs';
import * as ecsPatterns from '@aws-cdk/aws-ecs-patterns';
import * as logs from '@aws-cdk/aws-logs';
import * as waf from '@aws-cdk/aws-wafv2';
import * as route53 from '@aws-cdk/aws-route53';
import * as targets from '@aws-cdk/aws-route53-targets';
import { Config } from './config';
import { CommonStack } from './common-stack';
import { AcsDataStack } from './acs-data-stack';
import { EleriumDataStack } from './elerium-data-stack';
import { EleriumComputeStack } from './elerium-compute-stack';
import { Helper } from './resources/helper';

export interface ACSComputeProps {
  vpc: ec2.Vpc;
  commonData: CommonStack;
  cert: certificateManager.Certificate;
  acsData: AcsDataStack;
  eleriumData: EleriumDataStack;
  eleriumCompute: EleriumComputeStack;
}

export class ACSComputeStack extends cdk.Stack {
  // Stack Consts
  ADDON_SERVICE_URL = 'addon-service.overwolf.wtf';

  // Stack Assets
  cluster: ecs.Cluster;
  acsService: ecsPatterns.ApplicationLoadBalancedEc2Service;
  acsWaf: waf.CfnWebACL;

  constructor(scope: cdk.Construct, config: Config, props: ACSComputeProps) {
    super(scope, config.prefix + 'AcsCompute', { env: config.env });

    this.cluster = this.setupECSCluster(config, props);
    this.acsService = this.setupAddonServiceTask(config, props);
    this.acsWaf = this.setupAddonServiceWaf(config, props);
    this.setupAddonServiceCloudFront(config, props);
    this.setupCrashLogFileValidationServiceTask(config, props);
    this.setupAddonServiceDataManagerTask(config, props);

    props.acsData.assetsBucket.grantReadWrite(props.eleriumCompute.eleriumIamInstanceRole);
  }

  private setupAddonServiceWaf(config: Config, props: ACSComputeProps) {
    const blacklistedIpSet = new waf.CfnIPSet(this, 'BlacklistedIpAddresses', {
      addresses: [
        '54.78.88.106/32',
        '185.5.29.93/32',
        '114.41.72.196/32',
        '210.6.50.2/32',
        '119.247.197.98/32',
        '165.22.162.240/32',
        '112.119.52.102/32',
        '112.201.68.210/32',
        '117.20.68.72/32',
        '71.232.204.143/32',
        '176.199.210.59/32',
        '76.121.85.116/32',
        '176.57.191.101/32',
      ],
      ipAddressVersion: 'IPV4',
      scope: 'CLOUDFRONT',
      description: 'These users are blacklisted from using Addon Service due to malicious behavior in the past.',
      name: 'AddonService-Blacklisted-IPs',
    });

    const emptyStringRegex = new waf.CfnRegexPatternSet(this, 'EmptyStringRegex', {
      regularExpressionList: [],
      scope: 'CLOUDFRONT',
      description: 'Matches an empty string',
      name: 'Match-Empty-String',
    });

    const acsWaf = new waf.CfnWebACL(this, 'AddonServiceWaf', {
      defaultAction: { allow: {} },
      scope: 'CLOUDFRONT',
      visibilityConfig: {
        cloudWatchMetricsEnabled: true,
        metricName: 'AddonServiceWaf',
        sampledRequestsEnabled: true,
      },
      description:
        'Firewall for AddonService CloudFront to prevent malicious users from affecting service performance.',
      name: 'Addon-Service-Firewall',
      rules: [
        {
          action: { block: {} },
          name: 'CurseForgeBlacklistedIps',
          priority: 1,
          statement: {
            ipSetReferenceStatement: {
              arn: blacklistedIpSet.attrArn,
            },
          },
          visibilityConfig: {
            cloudWatchMetricsEnabled: true,
            metricName: 'CurseForgeBlacklistedIps',
            sampledRequestsEnabled: true,
          },
        },
        {
          action: { block: {} },
          name: 'BadUserAgents',
          priority: 2,
          statement: {
            orStatement: {
              statements: [
                {
                  byteMatchStatement: {
                    fieldToMatch: {
                      singleHeader: { name: 'user-agent' },
                    },
                    positionalConstraint: 'CONTAINS',
                    searchStringBase64: 'R3V6emxlSHR0cA==',
                    textTransformations: [{ priority: 0, type: 'NONE' }],
                  },
                },
                {
                  byteMatchStatement: {
                    fieldToMatch: {
                      singleHeader: { Name: 'user-agent' },
                    },
                    positionalConstraint: 'CONTAINS',
                    searchStringBase64: 'cHl0aG9uLXJlcXVlc3Rz',
                    textTransformations: [{ priority: 0, type: 'LOWERCASE' }],
                  },
                },
                {
                  byteMatchStatement: {
                    fieldToMatch: {
                      singleHeader: { name: 'user-agent' },
                    },
                    positionalConstraint: 'CONTAINS',
                    searchStringBase64: 'Q0YgQm90',
                    textTransformations: [{ priority: 0, type: 'NONE' }],
                  },
                },
              ],
            },
          },
          visibilityConfig: {
            cloudWatchMetricsEnabled: true,
            metricName: 'BadUserAgents',
            sampledRequestsEnabled: true,
          },
        },
        {
          action: { count: {} },
          name: 'LoggingRequestRule',
          priority: 3,
          statement: {
            rateBasedStatement: { aggregateKeyType: 'IP', limit: 4000 },
          },
          visibilityConfig: {
            cloudWatchMetricsEnabled: true,
            metricName: 'LoggingRequestRule',
            sampledRequestsEnabled: true,
          },
        },
        {
          action: { count: {} },
          name: 'MissingAuthHeaderTest',
          priority: 4,
          statement: {
            regexPatternSetReferenceStatement: {
              arn: emptyStringRegex.attrArn,
              fieldToMatch: { singleHeader: { name: 'authorization' } },
              textTransformations: [{ priority: 0, type: 'NONE' }],
            },
          },
          visibilityConfig: {
            cloudWatchMetricsEnabled: true,
            metricName: 'MissingAuthHeaderTest',
            sampledRequestsEnabled: true,
          },
        },
      ],
    });

    return acsWaf;
  }

  private generateAddonServiceBehavior(path: string): cloudfront.Behavior {
    return {
      allowedMethods: cloudfront.CloudFrontAllowedMethods.GET_HEAD_OPTIONS,
      compress: false,
      forwardedValues: {
        queryString: true,
        headers: ['Host', 'Origin'],
      },
      minTtl: cdk.Duration.seconds(0),
      maxTtl: cdk.Duration.seconds(15),
      defaultTtl: cdk.Duration.seconds(15),
      pathPattern: path,
    };
  }

  private setupAddonServiceCloudFront(config: Config, props: ACSComputeProps) {
    const distro = new cloudfront.CloudFrontWebDistribution(this, 'AddonServiceDist', {
      originConfigs: [
        {
          customOriginSource: {
            domainName: 'addon-service-internal.overwolf.wtf',
            originKeepaliveTimeout: cdk.Duration.seconds(10),
            originProtocolPolicy: cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
          },
          behaviors: [
            {
              allowedMethods: cloudfront.CloudFrontAllowedMethods.GET_HEAD_OPTIONS,
              compress: false,
              defaultTtl: cdk.Duration.minutes(20),
              forwardedValues: {
                queryString: true,
                headers: ['Host', 'Origin'],
                queryStringCacheKeys: ['supportsAddons'],
              },
              minTtl: cdk.Duration.seconds(0),
              pathPattern: '/api/v2/game',
            },
            {
              allowedMethods: cloudfront.CloudFrontAllowedMethods.GET_HEAD_OPTIONS,
              compress: false,
              defaultTtl: cdk.Duration.minutes(20),
              forwardedValues: {
                queryString: true,
                headers: ['Host', 'Origin'],
                queryStringCacheKeys: ['supportsAddons'],
              },
              minTtl: cdk.Duration.seconds(0),
              pathPattern: '/api/game',
            },
            {
              allowedMethods: cloudfront.CloudFrontAllowedMethods.GET_HEAD_OPTIONS,
              compress: false,
              defaultTtl: cdk.Duration.minutes(20),
              forwardedValues: {
                queryString: false,
                headers: ['Host', 'Origin'],
              },
              minTtl: cdk.Duration.seconds(0),
              pathPattern: '/api/v2/game/timestamp',
            },
            this.generateAddonServiceBehavior('/api/v2/minecraft/modloader'),
            this.generateAddonServiceBehavior('/api/v2/minecraft/version'),
            this.generateAddonServiceBehavior('/api/v2/minecraft/modloader/timestamp'),
            this.generateAddonServiceBehavior('/api/v2/minecraft/version/timestamp'),
            this.generateAddonServiceBehavior('/api/v2/category'),
            this.generateAddonServiceBehavior('/api/v2/category/timestamp'),
            {
              allowedMethods: cloudfront.CloudFrontAllowedMethods.ALL,
              compress: false,
              forwardedValues: {
                queryString: true,
                cookies: {
                  forward: 'all',
                },
                headers: ['*'],
              },
              minTtl: cdk.Duration.seconds(0),
              maxTtl: cdk.Duration.seconds(15),
              defaultTtl: cdk.Duration.seconds(15),
              isDefaultBehavior: true,
            },
          ],
        },
      ],
      viewerCertificate: cloudfront.ViewerCertificate.fromAcmCertificate(props.cert, {
        aliases: [this.ADDON_SERVICE_URL, 'addons-ecs.forgesvc.net'],
      }),
      priceClass: cloudfront.PriceClass.PRICE_CLASS_ALL,
      loggingConfig: {
        bucket: props.commonData.accessLogs,
        prefix: 'addon-service',
      },
      webACLId: this.acsWaf.attrArn,
    });

    // Assets DNS records
    new route53.ARecord(this, 'AcsDistARecord', {
      recordName: 'addon-service.overwolf.wtf',
      zone: props.commonData.internalZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distro)),
    });

    new route53.AaaaRecord(this, 'AcsDistAaaaRecord', {
      recordName: 'addon-service.overwolf.wtf',
      zone: props.commonData.internalZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distro)),
    });

    new route53.ARecord(this, 'AcsDistSvcZoneARecord', {
      recordName: 'addons-ecs.forgesvc.net',
      zone: props.commonData.servicesZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distro)),
    });

    new route53.AaaaRecord(this, 'AcsDistSvcZoneAaaaRecord', {
      recordName: 'addons-ecs.forgesvc.net',
      zone: props.commonData.servicesZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distro)),
    });

    return distro;
  }

  private setupECSCluster(config: Config, props: ACSComputeProps) {
    const cluster = new ecs.Cluster(this, 'AddonServiceCluster', {
      clusterName: 'AddonService',
      vpc: props.vpc,
    });

    cluster.addCapacity('AddonServiceClusterScalingCapacity', {
      instanceType: ec2.InstanceType.of(ec2.InstanceClass.C5, ec2.InstanceSize.XLARGE4),
      desiredCapacity: config.acsClusterCapacity,
    });

    return cluster;
  }

  private setupCrashLogFileValidationServiceTask(config: Config, props: ACSComputeProps) {
    const logGroup = new logs.LogGroup(this, 'AddonServiceCrashLogFileValidationServiceLogGroup', {
      logGroupName: 'AddonServiceCrashLogFileValidation',
      retention: logs.RetentionDays.SIX_MONTHS,
    });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'CrashLogFileValidationServiceTask');

    taskDefinition.addContainer('CrashLogFileValidationServiceContainer', {
      image: ecs.ContainerImage.fromEcrRepository(props.acsData.ecrCrashLogFileInvalidationService),
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'Task',
        logGroup,
      }),
      memoryLimitMiB: 2048,
      environment: {
        'CrashLogFileValidationService:AddonServiceAddress': 'https://' + this.ADDON_SERVICE_URL,
        'CrashLogFileValidationService:Env': config.envName,
        'CrashLogFileValidationService:QueueUrl': props.acsData.crashLogFileValidationQueue.queueUrl,
      },
      secrets: {
        'CrashLogFileValidationService:AddonServiceAPIKey': ecs.Secret.fromSecretsManager(
          props.acsData.addonServiceApiKey
        ),
      },
    });

    /* This service handles file invalidations for addons in the data manager */
    new ecs.Ec2Service(this, 'CrashLogFileValidationService', {
      cluster: this.cluster,
      taskDefinition,
      desiredCount: 2,
    });

    // Permissions
    logGroup.grantWrite(taskDefinition.taskRole);
    props.acsData.crashLoggingS3.grantReadWrite(taskDefinition.taskRole);
    props.acsData.crashLogFileValidationQueue.grantConsumeMessages(taskDefinition.taskRole);
  }

  private setupAddonServiceDataManagerTask(config: Config, props: ACSComputeProps) {
    const logGroup = new logs.LogGroup(this, 'AddonServiceDataManagerLogGroup', {
      logGroupName: 'AddonServiceDataManager',
      retention: logs.RetentionDays.SIX_MONTHS,
    });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'DataManagerTask');

    taskDefinition.addContainer('DataManagerContainer', {
      image: ecs.ContainerImage.fromEcrRepository(props.acsData.ecrDataManager),
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'Task',
        logGroup,
      }),
      memoryLimitMiB: 2048,
      environment: {
        'DataManager:ElasticSearchAddress': 'https://' + props.acsData.dataManagerESCluster.attrDomainEndpoint,
        'DataManager:EleriumBaseDomain': 'https://' + props.eleriumCompute.eleriumControlDomain.domainName,
        'DataManager:Env': config.envName,
        'DataManager:QueueUrl': props.acsData.dataManagerQueue.queueUrl,
        'DataManager:RedisUrl':
          props.acsData.dataManagerRedisCluster.attrPrimaryEndPointAddress +
          ':' +
          props.acsData.dataManagerRedisCluster.attrPrimaryEndPointPort,
        'DataManager:SqsUrl': 'https://sqs.' + config.env.region + '.amazonaws.com',
      },
      secrets: {
        'DataManager:EleriumApiKey': ecs.Secret.fromSecretsManager(props.acsData.eleriumApiKey),
      },
    });

    /* This is the data manager service for the addon service */
    const dataManagerService = new ecs.Ec2Service(this, 'DataManager', {
      cluster: this.cluster,
      taskDefinition,
      desiredCount: 20,
    });

    // Permissions
    logGroup.grantWrite(taskDefinition.taskRole);
    props.acsData.dataManagerQueue.grantConsumeMessages(taskDefinition.taskRole);
    props.acsData.dataManagerRedisCluster.addIngressPeer(dataManagerService.cluster.connections, true);
  }

  private setupAddonServiceTask(config: Config, props: ACSComputeProps) {
    const logGroup = new logs.LogGroup(this, 'AddonServiceLogGroup', {
      logGroupName: 'AddonService',
      retention: logs.RetentionDays.SIX_MONTHS,
    });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'AddonServiceTask');

    const container = taskDefinition.addContainer('AddonServiceContainer', {
      image: ecs.ContainerImage.fromEcrRepository(props.acsData.ecrAddonService),
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'Task',
        logGroup,
      }),
      memoryLimitMiB: 12288,
      environment: {
        ASPNETCORE_ENVIRONMENT: config.envName,
        ELASTICSEARCH_URL: 'https://' + props.acsData.dataManagerESCluster.attrDomainEndpoint,
        REDIS_URL: props.acsData.addonServiceRedisCluster.attrPrimaryEndPointAddress,
      },
      secrets: {
        APIKEY: ecs.Secret.fromSecretsManager(props.acsData.apiKey),
        AUTHENTICATION_ID: ecs.Secret.fromSecretsManager(props.acsData.authenticationID),
        AUTHENTICATION_KEY: ecs.Secret.fromSecretsManager(props.acsData.authenticationKey),
        ENCRYPTION_TOKEN_KEY: ecs.Secret.fromSecretsManager(props.acsData.encryptionTokenKey),
        GAME_MANAGEMENT_APIKEY: ecs.Secret.fromSecretsManager(props.acsData.gameManagementApiKey),
        IN_API_KEY: ecs.Secret.fromSecretsManager(props.acsData.inApiKey),
        LOGSERVICE_APIKEY: ecs.Secret.fromSecretsManager(props.acsData.logServiceApiKey),
        RDS_DEFAULT_HOSTNAME: ecs.Secret.fromSecretsManager(props.eleriumData.db.secret!, 'host'),
        RDS_DEFAULT_PASSWORD: ecs.Secret.fromSecretsManager(props.acsData.eleriumDatabaseSecret, 'password'),
        RDS_DEFAULT_USERNAME: ecs.Secret.fromSecretsManager(props.acsData.eleriumDatabaseSecret, 'username'),
      },
    });

    container.addPortMappings({ containerPort: 80 });

    /* Addon Service */
    const addonServiceService = new ecsPatterns.ApplicationLoadBalancedEc2Service(this, 'AddonService', {
      cluster: this.cluster,
      taskDefinition,
      desiredCount: 20,
      certificate: props.cert,
      domainZone: props.commonData.internalZone,
      domainName: 'addon-service-internal.overwolf.wtf',
    });

    // Permissions
    logGroup.grantWrite(taskDefinition.taskRole);
    props.acsData.crashLoggingS3.grantReadWrite(taskDefinition.taskRole);
    props.acsData.addonServiceRedisCluster.addIngressPeer(addonServiceService.cluster.connections, true);
    props.acsData.crashLogDynamoTable.grantReadWriteData(taskDefinition.taskRole);
    Helper.addP2PDefaultPortIngress(props.eleriumData.db.connections, addonServiceService.cluster.connections);

    return addonServiceService;
  }
}
