// tslint:disable-next-line: no-submodule-imports
import 'source-map-support/register';

import * as cdk from '@aws-cdk/core';
import { CommonStack } from './common-stack';
import { VpcStack } from './vpc-stack';
import { Config, CONFIGS } from './config';
import { DlsDataStack } from './dls-data-stack';
import { CommonComputeStack } from './common-compute-stack';
import { DlsComputeStack } from './dls-compute-stack';
import { GtaGamingStack } from './gta-gaming-stack';
import { BukkitForumsDataStack } from './bukkit-forums-data-stack';
import { BukkitForumsComputeStack } from './bukkit-forums-compute-stack';
import { EleriumDataStack } from './elerium-data-stack';
import { EleriumComputeStack } from './elerium-compute-stack';
import { AcsDataStack } from './acs-data-stack';
import { BastionStack } from './bastion-stack';
import { SentryDataStack } from './sentry-data-stack';
import { SentryComputeStack } from './sentry-compute-stack';
import { ClamAvDataStack } from './clamav-data-stack';
import { ClamAvComputeStack } from './clamav-compute-stack';
import { HyruleDataStack } from './hyrule-data-stack';
import { HyruleComputeStack } from './hyrule-compute-stack';
import { RepoHostComputeStack } from './repo-host-compute-stack';
import { RepoHostDataStack } from './repo-host-data-stack';
import { LegacyAcsComputeStack } from './legacy-acs-compute-stack';
import { LegacyAcsDataStack } from './legacy-acs-data-stack';
import { ACSComputeStack } from './acs-compute-stack';
import { CertificatesStack } from './certificates-stack';
import { PackagerComputeStack } from './packager-compute-stack';
import { FilesServiceComputeStack } from './files-service-compute-stack';
import { FilesServiceDataStack } from './files-service-data-stack';
import { CfWidgetDataStack } from './cfwidget-data-stack';
import { CFWidgetComputeStack } from './cfwidget-compute-stack';

function createStacks(scope: cdk.App, config: Config) {
  // Common infra
  const vpcStack = new VpcStack(scope, config);
  const commonStack = new CommonStack(scope, config);
  const commonComputeStack = new CommonComputeStack(scope, config, { vpc: vpcStack.vpc, commonData: commonStack });
  const certsStack = new CertificatesStack(scope, config, { commonStack });

  // Download Service
  const dlsDataStack = new DlsDataStack(scope, config, { vpc: vpcStack.vpc, commonComputeStack });
  new DlsComputeStack(scope, config, {
    cluster: commonComputeStack.cluster,
    dataStack: dlsDataStack,
    cert: certsStack.cert,
    commonStack,
    vpc: vpcStack.vpc,
  });

  // Elerium
  const eleriumDataStack = new EleriumDataStack(scope, config, {
    vpc: vpcStack.vpc,
    cert: certsStack.cert,
    commonStack,
    backupsS3: commonStack.backupStorage,
  });
  const eleriumComputeStack = new EleriumComputeStack(scope, config, {
    vpc: vpcStack.vpc,
    cert: certsStack.cert,
    dataStack: eleriumDataStack,
    commonStack,
  });

  // Addon Client Service
  const acsDataStack = new AcsDataStack(scope, config, {
    cert: certsStack.cert,
    commonStack,
    eleriumDataStack,
    vpc: vpcStack.vpc,
    commonComputeStack,
    eleriumComputeStack,
  });
  new ACSComputeStack(scope, config, {
    vpc: vpcStack.vpc,
    commonData: commonStack,
    acsData: acsDataStack,
    eleriumData: eleriumDataStack,
    cert: certsStack.cert,
    eleriumCompute: eleriumComputeStack,
  });

  // GTA Gaming
  new GtaGamingStack(scope, config, { commonStack });

  // Bukkit Forums
  const bukkitDataStack = new BukkitForumsDataStack(scope, config, {
    vpc: vpcStack.vpc,
  });
  new BukkitForumsComputeStack(scope, config, {
    vpc: vpcStack.vpc,
    cert: certsStack.cert,
    commonStack,
    dataStack: bukkitDataStack,
  });

  // ClamAV
  const clamAvDataStack = new ClamAvDataStack(scope, config);
  new ClamAvComputeStack(scope, config, {
    dataStack: clamAvDataStack,
    cluster: commonComputeStack.cluster,
    zone: commonStack.internalZone,
  });

  // Sentry
  const sentryDataStack = new SentryDataStack(scope, config, {
    vpc: vpcStack.vpc,
  });

  new SentryComputeStack(scope, config, {
    commonCompute: commonComputeStack,
    commonStack,
    dataStack: sentryDataStack,
    cert: certsStack.cert,
    vpc: vpcStack.vpc,
  });

  // Hyrule
  const hyruleDataStack = new HyruleDataStack(scope, config, {
    vpc: vpcStack.vpc,
  });

  new HyruleComputeStack(scope, config, {
    commonCompute: commonComputeStack,
    commonStack,
    dataStack: hyruleDataStack,
    cert: certsStack.cert,
    vpc: vpcStack.vpc,
  });

  // Repo Host
  const repoHostDataStack = new RepoHostDataStack(scope, config, { vpc: vpcStack.vpc, commonComputeStack });
  new RepoHostComputeStack(scope, config, {
    vpc: vpcStack.vpc,
    cluster: commonComputeStack.cluster,
    dataStack: repoHostDataStack,
    commonStack,
    cert: certsStack.cert,
  });

  // Legacy ACS
  const legacyAcsDataStack = new LegacyAcsDataStack(scope, config, {
    vpc: vpcStack.vpc,
    cert: certsStack.cert,
    commonStack,
    eleriumDataStack,
  });

  new LegacyAcsComputeStack(scope, config, {
    vpc: vpcStack.vpc,
    cert: certsStack.cert,
    dataStack: legacyAcsDataStack,
    eleriumDataStack,
    commonStack,
  });

  // Packager
  new PackagerComputeStack(scope, config, {
    eleriumDataStack,
    commonStack,
    cert: certsStack.cert,
    vpc: vpcStack.vpc,
  });

  // Files Service
  const filesServiceDataStack = new FilesServiceDataStack(scope, config);
  new FilesServiceComputeStack(scope, config, {
    dataStack: filesServiceDataStack,
    commonStack,
    cluster: commonComputeStack.cluster,
    cert: certsStack.cert,
    eleriumDataStack,
  });

  // CF Widget
  const cfWidgetDataStack = new CfWidgetDataStack(scope, config);
  new CFWidgetComputeStack(scope, config, {
    cert: certsStack.cert,
    cluster: commonComputeStack.cluster,
    commonStack,
    dataStack: cfWidgetDataStack,
    eleriumDataStack,
  });

  // Bastion
  new BastionStack(scope, config, {
    vpcStack,
    commonStack,
    acsDataStack,
    bukkitDataStack,
    eleriumDataStack,
    sentryDataStack,
    hyruleDataStack,
    repoHostDataStack,
    dlsDataStack,
    legacyAcsDataStack,
  });
}

const app = new cdk.App();
for (const config of CONFIGS) {
  createStacks(app, config);
}
