import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import {
  Config,
  BUKKIT_LEGACY_DATA_BUCKET,
  ELERIUM_LEGACY_DATA_BUCKET,
  ELERIUM_LEGACY_ASSETS_BUCKET,
  ACS_LEGACY_ASSETS_BUCKET,
  ACS_CRASH_LOGS_LEGACY_ASSETS_BUCKET,
  ACS_MOD_LOADERS_LEGACY_ASSETS_BUCKET,
} from './config';
import { CommonStack } from './common-stack';
import { AcsDataStack } from './acs-data-stack';
import { EleriumDataStack } from './elerium-data-stack';
import { BukkitForumsDataStack } from './bukkit-forums-data-stack';
import { VpcStack } from './vpc-stack';
import { BastionSecurityGroupStack } from './bastion-sg-stack';
import { SentryDataStack } from './sentry-data-stack';
import { HyruleDataStack } from './hyrule-data-stack';
import { RepoHostDataStack } from './repo-host-data-stack';
import { DlsDataStack } from './dls-data-stack';
import { LegacyAcsDataStack } from './legacy-acs-data-stack';

export interface BastionProps {
  vpcStack: VpcStack;
  commonStack: CommonStack;
  acsDataStack: AcsDataStack;
  eleriumDataStack: EleriumDataStack;
  bukkitDataStack: BukkitForumsDataStack;
  sentryDataStack: SentryDataStack;
  hyruleDataStack: HyruleDataStack;
  repoHostDataStack: RepoHostDataStack;
  dlsDataStack: DlsDataStack;
  legacyAcsDataStack: LegacyAcsDataStack;
}

const windowsAMI = ec2.MachineImage.latestWindows(
  ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_CONTAINERSLATEST
);

/**
 * The Bastion stack manages a bastion host that grants access to the production environment.
 */
export class BastionStack extends cdk.Stack {
  bastion: ec2.BastionHostLinux;
  windowsBastion: ec2.Instance;

  constructor(scope: cdk.Construct, config: Config, props: BastionProps) {
    super(scope, config.prefix + 'Bastion', { env: config.env });

    const sg = new BastionSecurityGroupStack(this, config, { vpc: props.vpcStack.vpc }).securityGroup;

    // Create the Bastion
    this.bastion = new ec2.BastionHostLinux(this, 'BastionHost', {
      vpc: props.vpcStack.vpc,
      subnetSelection: { subnetType: ec2.SubnetType.PUBLIC }, // Allows access to resources outside the account for syncing
      instanceType: ec2.InstanceType.of(ec2.InstanceClass.M5A, ec2.InstanceSize.XLARGE2),
      blockDevices: [
        {
          deviceName: '/dev/xvda',
          volume: ec2.BlockDeviceVolume.ebs(100, {
            encrypted: true,
          }),
        },
      ],
      // Security group lives in its own stack to avoid cyclic references
      securityGroup: sg,
    });

    this.windowsBastion = new ec2.Instance(this, 'WindowsInstance', {
      instanceType: ec2.InstanceType.of(ec2.InstanceClass.M5A, ec2.InstanceSize.XLARGE2),
      vpc: props.vpcStack.vpc,
      vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
      machineImage: windowsAMI,
      blockDevices: [
        {
          deviceName: '/dev/xvda',
          volume: ec2.BlockDeviceVolume.ebs(100, {
            encrypted: true,
          }),
        },
      ],
      instanceName: 'Windows Bastion',
      securityGroup: sg,
      keyName: 'windows-bastion',
    });

    // Give the Bastion a static IP
    new ec2.CfnEIP(this, 'EIP', {
      instanceId: this.bastion.instanceId,
    });

    // Give the Windows Bastion a static IP
    new ec2.CfnEIP(this, 'WindowsEIP', {
      instanceId: this.windowsBastion.instanceId,
    });

    // Allow SSH from Amazon VPN
    this.bastion.allowSshAccessFrom(ec2.Peer.prefixList(config.prefixList));

    // Grant the Bastion access to legacy buckets
    this.bastion.role.attachInlinePolicy(
      new iam.Policy(this, 'BukkitS3Policy', {
        statements: [
          new iam.PolicyStatement({
            actions: ['s3:ListBucket'],
            resources: [
              ACS_LEGACY_ASSETS_BUCKET,
              BUKKIT_LEGACY_DATA_BUCKET,
              ELERIUM_LEGACY_ASSETS_BUCKET,
              ELERIUM_LEGACY_DATA_BUCKET,
              ACS_CRASH_LOGS_LEGACY_ASSETS_BUCKET,
              ACS_MOD_LOADERS_LEGACY_ASSETS_BUCKET,
            ],
          }),
          new iam.PolicyStatement({
            actions: ['s3:GetObject', 's3:CopyObject'],
            resources: [
              ACS_LEGACY_ASSETS_BUCKET + '/*',
              BUKKIT_LEGACY_DATA_BUCKET + '/*',
              ELERIUM_LEGACY_ASSETS_BUCKET + '/*',
              ACS_MOD_LOADERS_LEGACY_ASSETS_BUCKET + '/*',
              ACS_CRASH_LOGS_LEGACY_ASSETS_BUCKET + '/*',
              ELERIUM_LEGACY_DATA_BUCKET + '/Overwolf/*',
              ELERIUM_LEGACY_DATA_BUCKET + '/repohost/*', // TODO: TMP
            ],
          }),
        ],
      })
    );

    // Grant permissions to Common resources
    props.commonStack.backupStorage.grantReadWrite(this.bastion.role);

    // Grant access to Addon Client Service resources
    props.acsDataStack.assetsBucket.grantReadWrite(this.bastion.role);
    props.acsDataStack.crashLoggingS3.grantReadWrite(this.bastion.role);
    props.legacyAcsDataStack.modloadersAssetsBuckets.grantReadWrite(this.bastion.role);

    // Grant access to Elerium resources
    props.eleriumDataStack.assetsBucket.grantReadWrite(this.bastion.role);
    props.eleriumDataStack.db.connections.allowDefaultPortFrom(this.bastion);

    // Grant access to Bukkit resources
    props.bukkitDataStack.efsBukkitForums.connections.allowDefaultPortFrom(this.bastion);
    props.bukkitDataStack.databaseCluster.connections.allowDefaultPortFrom(this.bastion);

    // Grant access to Hyrule resources
    props.hyruleDataStack.databaseCluster.connections.allowDefaultPortFrom(this.bastion);

    // Sentry Resources
    props.sentryDataStack.ecrSentry.grantPullPush(this.bastion.role);
    props.sentryDataStack.databaseCluster.connections.allowDefaultPortFrom(this.bastion);
    props.sentryDataStack.redis.addIngressPeer(this.bastion.connections);

    // RepoHost Resources
    props.repoHostDataStack.efsRepoHost.connections.allowDefaultPortFrom(this.bastion);

    // DLS
    props.dlsDataStack.redis.addIngressPeer(this.bastion.connections);

    // Allow SSH from Amazon VPN
    this.windowsBastion.connections.allowFrom(ec2.Peer.prefixList(config.prefixList), ec2.Port.tcp(3389));

    // Grant the Bastion access to legacy buckets
    this.windowsBastion.role.attachInlinePolicy(
      new iam.Policy(this, 'WindowsBukkitS3Policy', {
        statements: [
          new iam.PolicyStatement({
            actions: ['s3:ListBucket'],
            resources: [
              ACS_LEGACY_ASSETS_BUCKET,
              BUKKIT_LEGACY_DATA_BUCKET,
              ELERIUM_LEGACY_ASSETS_BUCKET,
              ELERIUM_LEGACY_DATA_BUCKET,
              ACS_CRASH_LOGS_LEGACY_ASSETS_BUCKET,
              ACS_MOD_LOADERS_LEGACY_ASSETS_BUCKET,
            ],
          }),
          new iam.PolicyStatement({
            actions: ['s3:GetObject', 's3:CopyObject'],
            resources: [
              ACS_LEGACY_ASSETS_BUCKET + '/*',
              BUKKIT_LEGACY_DATA_BUCKET + '/*',
              ELERIUM_LEGACY_ASSETS_BUCKET + '/*',
              ACS_MOD_LOADERS_LEGACY_ASSETS_BUCKET + '/*',
              ACS_CRASH_LOGS_LEGACY_ASSETS_BUCKET + '/*',
              ELERIUM_LEGACY_DATA_BUCKET + '/Overwolf/*',
              ELERIUM_LEGACY_DATA_BUCKET + '/repohost/*', // TODO: TMP
            ],
          }),
        ],
      })
    );

    // Grant permissions to Common resources
    props.commonStack.backupStorage.grantReadWrite(this.windowsBastion.role);

    // Grant access to Addon Client Service resources
    props.acsDataStack.assetsBucket.grantReadWrite(this.windowsBastion.role);
    props.acsDataStack.crashLoggingS3.grantReadWrite(this.windowsBastion.role);
    props.legacyAcsDataStack.modloadersAssetsBuckets.grantReadWrite(this.windowsBastion.role);

    // Grant access to Elerium resources
    props.eleriumDataStack.assetsBucket.grantReadWrite(this.windowsBastion.role);
    props.eleriumDataStack.db.connections.allowDefaultPortFrom(this.windowsBastion);

    // Grant access to Bukkit resources
    props.bukkitDataStack.efsBukkitForums.connections.allowDefaultPortFrom(this.windowsBastion);
    props.bukkitDataStack.databaseCluster.connections.allowDefaultPortFrom(this.windowsBastion);

    // Grant access to Hyrule resources
    props.hyruleDataStack.databaseCluster.connections.allowDefaultPortFrom(this.windowsBastion);

    // Sentry Resources
    props.sentryDataStack.ecrSentry.grantPullPush(this.windowsBastion.role);
    props.sentryDataStack.databaseCluster.connections.allowDefaultPortFrom(this.windowsBastion);
    props.sentryDataStack.redis.addIngressPeer(this.windowsBastion.connections);

    // RepoHost Resources
    props.repoHostDataStack.efsRepoHost.connections.allowDefaultPortFrom(this.windowsBastion);

    // DLS
    props.dlsDataStack.redis.addIngressPeer(this.windowsBastion.connections);
  }
}
