import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as ecs from '@aws-cdk/aws-ecs';
import * as ecsPatterns from '@aws-cdk/aws-ecs-patterns';
import * as elasticbeanstalk from '@aws-cdk/aws-elasticbeanstalk';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as route53 from '@aws-cdk/aws-route53';
import { Config } from './config';
import { CommonStack } from './common-stack';
import { BukkitForumsDataStack } from './bukkit-forums-data-stack';
import { Helper } from './resources/helper';

interface BukkitForumsComputeStackProps {
  vpc: ec2.Vpc;
  cert: certificateManager.Certificate;
  commonStack: CommonStack;
  dataStack: BukkitForumsDataStack;
}
/**
 * The Bukkit Forums - Compute
 */
export class BukkitForumsComputeStack extends cdk.Stack {
  constructor(scope: cdk.Construct, config: Config, props: BukkitForumsComputeStackProps) {
    super(scope, config.prefix + 'BukkitForumsCompute', { env: config.env });

    // Production Forum
    const forumsIamRole = new iam.Role(this, 'BukkitForumsEnvServiceRole', {
      assumedBy: new iam.ServicePrincipal('elasticbeanstalk.amazonaws.com'),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSElasticBeanstalkService'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSElasticBeanstalkEnhancedHealth'),
      ],
    });

    const forumsIamInstanceRole = new iam.Role(this, 'BukkitForumsEnvInstanceRole', {
      assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkWebTier'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkMulticontainerDocker'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkWorkerTier'),
      ],
    });

    const forumsIamInstanceProfile = new iam.CfnInstanceProfile(this, 'BukkitForumsEnvInstanceRoleProfile', {
      roles: [forumsIamInstanceRole.roleName],
    });

    forumsIamInstanceProfile.node.addDependency(forumsIamInstanceRole);

    const forumsSecurityGroup = new ec2.SecurityGroup(this, 'BukkitForumsSecurityGroup', {
      vpc: props.vpc,
    });

    const app = new elasticbeanstalk.CfnApplication(this, 'BeanstalkApp', {
      applicationName: 'bukkit-forums',
      resourceLifecycleConfig: {
        serviceRole: forumsIamRole.roleArn,
      },
    });

    app.node.addDependency(forumsIamRole);

    const env = new elasticbeanstalk.CfnEnvironment(this, 'BeanstalkEnv', {
      applicationName: app.applicationName!,
      solutionStackName: '64bit Amazon Linux 2018.03 v2.9.8 running PHP 5.5',
      optionSettings: [
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'InstanceType',
          value: 'c5.xlarge',
        },
        {
          namespace: 'aws:autoscaling:asg',
          optionName: 'MinSize',
          value: config.bukkitElasticBeanstalkCount.toString(),
        },
        {
          namespace: 'aws:autoscaling:asg',
          optionName: 'MaxSize',
          value: '3',
        },
        {
          namespace: 'aws:elasticbeanstalk:cloudwatch:logs',
          optionName: 'StreamLogs',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:cloudwatch:logs',
          optionName: 'RetentionInDays',
          value: '365',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'ALLOW_ADMIN_URL',
          value: 'false',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'COOKIE_DOMAIN',
          value: '.bukkit.org',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'PHP_ENV',
          value: 'production',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'EFS_ID',
          value: props.dataStack.efsBukkitForums.fileSystemId,
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_SERVER',
          value: props.dataStack.databaseCluster.clusterEndpoint.hostname,
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_USER',
          value: props.dataStack.databaseSecret.secretValueFromJson('username').toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_PASSWORD',
          value: props.dataStack.databaseSecret.secretValueFromJson('password').toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:environment',
          optionName: 'LoadBalancerType',
          value: 'network',
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'ELBScheme',
          value: 'internal',
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'VPCId',
          value: props.vpc.vpcId,
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'Subnets',
          value: props.vpc.privateSubnets.map((s) => s.subnetId).join(','),
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'ELBSubnets',
          value: props.vpc.privateSubnets.map((s) => s.subnetId).join(','),
        },
        {
          namespace: 'aws:elasticbeanstalk:environment',
          optionName: 'ServiceRole',
          value: forumsIamRole.roleName,
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'IamInstanceProfile',
          value: forumsIamInstanceProfile.ref, // Ref on an instance profile is the name
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'DeploymentPolicy',
          value: 'Rolling',
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'BatchSizeType',
          value: 'Fixed',
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'BatchSize',
          value: '1',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'SecurityGroups',
          value: forumsSecurityGroup.securityGroupName,
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'RollingUpdateEnabled',
          value: 'true',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'MaxBatchSize',
          value: '1',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'MinInstancesInService',
          value: '1',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'RollingUpdateType',
          value: 'Health',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'Protocol',
          value: 'TCP',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'DeregistrationDelay',
          value: '10',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckInterval',
          value: '10',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthyThresholdCount',
          value: config.bukkitElasticBeanstalkCount.toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'UnhealthyThresholdCount',
          value: config.bukkitElasticBeanstalkCount.toString(),
        },
      ],
    });

    env.addDependsOn(app);
    env.node.addDependency(forumsIamRole);
    env.node.addDependency(forumsIamInstanceProfile);
    env.node.addDependency(forumsSecurityGroup);

    Helper.addDefaultPortIngress(props.dataStack.efsBukkitForums.connections, forumsSecurityGroup);
    Helper.addDefaultPortIngress(props.dataStack.databaseCluster.connections, forumsSecurityGroup);

    const varnishCluster = new ecs.Cluster(this, 'VarnishCluster', {
      vpc: props.vpc,
      capacity: {
        instanceType: ec2.InstanceType.of(ec2.InstanceClass.C5, ec2.InstanceSize.XLARGE),
        desiredCapacity: 3,
      },
    });

    const varnishService = new ecsPatterns.ApplicationLoadBalancedEc2Service(this, 'VarnishService', {
      cluster: varnishCluster,
      desiredCount: 3,
      memoryLimitMiB: 2048,
      minHealthyPercent: 100,
      maxHealthyPercent: 200,
      taskImageOptions: {
        containerPort: 8080,
        image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrImitari, 'prod-6fc3bf8'),
        logDriver: new ecs.AwsLogDriver({
          streamPrefix: 'varnish-service',
          logRetention: 30,
        }),
        environment: {
          IMITARI_SITE_HOST: env.getAtt('EndpointURL').toString(),
          IMITARI_STAGE: 'production',
          VARNISH_CACHE_SIZE: '768',
          VARNISH_TTL: '86400',
        },
      },
      certificate: props.cert,
      domainName: 'bukkit-forums.overwolf.wtf', // Update COOKIE_DOMAIN above
      domainZone: props.commonStack.internalZone,
    });

    varnishService.loadBalancer.logAccessLogs(props.commonStack.accessLogs, 'imitari');
    varnishService.targetGroup.enableCookieStickiness(cdk.Duration.days(1));
    varnishService.targetGroup.configureHealthCheck({
      path: '/debug/running',
    });

    for (const sg of varnishService.cluster.connections.securityGroups) {
      forumsSecurityGroup.addIngressRule(sg, ec2.Port.tcp(80));
    }

    varnishService.node.addDependency(env);

    // Admin Forum
    const adminIamRole = new iam.Role(this, 'AdminBukkitForumsEnvServiceRole', {
      assumedBy: new iam.ServicePrincipal('elasticbeanstalk.amazonaws.com'),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSElasticBeanstalkService'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSElasticBeanstalkEnhancedHealth'),
      ],
    });

    const adminIamInstanceRole = new iam.Role(this, 'AdminBukkitForumsEnvInstanceRole', {
      assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkWebTier'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkMulticontainerDocker'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkWorkerTier'),
      ],
    });

    const adminIamInstanceProfile = new iam.CfnInstanceProfile(this, 'AdminBukkitForumsEnvInstanceRoleProfile', {
      roles: [forumsIamInstanceRole.roleName],
    });

    forumsIamInstanceProfile.node.addDependency(forumsIamInstanceRole);

    const adminEbSecurityGroup = new ec2.SecurityGroup(this, 'AdminBukkitForumsBeanstalkSecurityGroup', {
      vpc: props.vpc,
    });

    const adminLbSecurityGroup = new ec2.SecurityGroup(this, 'AdminBukkitForumsLoadBalancerSecurityGroup', {
      vpc: props.vpc,
    });

    const adminApp = new elasticbeanstalk.CfnApplication(this, 'AdminBeanstalkApp', {
      applicationName: 'bukkit-admin-forums',
      resourceLifecycleConfig: {
        serviceRole: adminIamRole.roleArn,
      },
    });

    const adminEnv = new elasticbeanstalk.CfnEnvironment(this, 'AdminBeanstalkEnv', {
      applicationName: adminApp.applicationName!,
      solutionStackName: '64bit Amazon Linux 2018.03 v2.9.8 running PHP 5.5',
      optionSettings: [
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'InstanceType',
          value: 'c5.xlarge',
        },
        {
          namespace: 'aws:autoscaling:asg',
          optionName: 'MinSize',
          value: '1',
        },
        {
          namespace: 'aws:autoscaling:asg',
          optionName: 'MaxSize',
          value: '1',
        },
        {
          namespace: 'aws:elasticbeanstalk:cloudwatch:logs',
          optionName: 'StreamLogs',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:cloudwatch:logs',
          optionName: 'RetentionInDays',
          value: '365',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'ALLOW_ADMIN_URL',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'COOKIE_DOMAIN',
          value: '.bukkit-forums-admin.overwolf.wtf',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'PHP_ENV',
          value: 'development',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'EFS_ID',
          value: props.dataStack.efsBukkitForums.fileSystemId,
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_SERVER',
          value: props.dataStack.databaseCluster.clusterEndpoint.hostname,
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_USER',
          value: props.dataStack.databaseSecret.secretValueFromJson('username').toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_PASSWORD',
          value: props.dataStack.databaseSecret.secretValueFromJson('password').toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:environment',
          optionName: 'LoadBalancerType',
          value: 'application',
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'VPCId',
          value: props.vpc.vpcId,
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'Subnets',
          value: props.vpc.privateSubnets.map((s) => s.subnetId).join(','),
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'ELBSubnets',
          value: props.vpc.publicSubnets.map((s) => s.subnetId).join(','),
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'AccessLogsS3Bucket',
          value: props.commonStack.accessLogs.bucketName,
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'AccessLogsS3Enabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment',
          optionName: 'ServiceRole',
          value: adminIamRole.roleName,
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'IamInstanceProfile',
          value: adminIamInstanceRole.roleName,
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'IamInstanceProfile',
          value: adminIamInstanceProfile.ref, // Ref on an instance profile is the name
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'SecurityGroups',
          value: adminEbSecurityGroup.securityGroupName,
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'SecurityGroups',
          value: adminLbSecurityGroup.securityGroupName,
        },
        {
          namespace: 'aws:elbv2:listener:443',
          optionName: 'Protocol',
          value: 'HTTPS',
        },
        {
          namespace: 'aws:elbv2:listener:443',
          optionName: 'SSLCertificateArns',
          value: props.cert.certificateArn,
        },
      ],
    });

    new route53.CnameRecord(this, 'AdminBukkitCname', {
      domainName: adminEnv.attrEndpointUrl,
      zone: props.commonStack.internalZone,
      recordName: 'bukkit-forums-admin',
    });

    adminEnv.addDependsOn(adminApp);
    adminEnv.node.addDependency(adminIamRole);
    adminEnv.node.addDependency(adminIamInstanceProfile);
    adminEnv.node.addDependency(adminEbSecurityGroup);
    adminEnv.node.addDependency(adminLbSecurityGroup);

    adminLbSecurityGroup.connections.allowFrom(ec2.Peer.prefixList(config.prefixList), ec2.Port.tcp(443));
    Helper.addDefaultPortIngress(props.dataStack.efsBukkitForums.connections, adminEbSecurityGroup);
    Helper.addDefaultPortIngress(props.dataStack.databaseCluster.connections, adminEbSecurityGroup);
  }
}
