import * as cdk from '@aws-cdk/core';
import * as rds from '@aws-cdk/aws-rds';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as efs from '@aws-cdk/aws-efs';
import * as ecr from '@aws-cdk/aws-ecr';
import * as iam from '@aws-cdk/aws-iam';
import { Config } from './config';
import { ElasticSearch } from './resources/elastic-search';

interface BukkitForumsDataStackProps {
  vpc: ec2.Vpc;
}
/**
 * The Bukkit Forums - Data
 */
export class BukkitForumsDataStack extends cdk.Stack {
  databaseCluster: rds.DatabaseCluster;
  databaseSecret: rds.DatabaseSecret;
  efsBukkitForums: efs.FileSystem;
  ecrImitari: ecr.Repository;

  constructor(scope: cdk.Construct, config: Config, props: BukkitForumsDataStackProps) {
    super(scope, config.prefix + 'BukkitForumsData', { env: config.env });

    // Database
    this.databaseCluster = new rds.DatabaseCluster(this, 'DatabaseCluster', {
      engine: rds.DatabaseClusterEngine.aurora({
        version: rds.AuroraEngineVersion.VER_1_22_2,
      }),
      instanceProps: {
        vpc: props.vpc,
        instanceType: ec2.InstanceType.of(ec2.InstanceClass.R4, ec2.InstanceSize.LARGE),
      },
      masterUser: { username: 'admin' },
      instances: config.bukkitAuroraInstanceCount,
    });

    this.databaseSecret = new rds.DatabaseSecret(this, 'BukkitDBApplicationSecret', {
      username: 'forums_bukkit',
      masterSecret: this.databaseCluster.secret,
    });

    // File Storage
    this.efsBukkitForums = new efs.FileSystem(this, 'FileSystem', {
      vpc: props.vpc,
      encrypted: true,
    });

    this.efsBukkitForums.connections.allowDefaultPortFrom(
      ec2.SecurityGroup.fromSecurityGroupId(this, 'DataSyncSecurityGroup', config.dataSyncSecurityGroup)
    );

    // Elastic Search
    // This is not an enviroment variable but is set inside the settings of the forums software.
    new ElasticSearch(this, 'ElasticSearch', config, {
      elasticsearchVersion: '5.6',
      elasticsearchClusterConfig: {
        instanceType: 'r5.large.elasticsearch',
        instanceCount: config.bukkitElasticSearchCount,
        zoneAwarenessEnabled: true,
        zoneAwarenessConfig: {
          availabilityZoneCount: 3,
        },
      },
      ebsOptions: {
        ebsEnabled: true,
        volumeSize: 100,
      },
      accessPolicies: new iam.PolicyDocument({
        // This is a VPC Cluster, don't do this for public clusters
        statements: [
          new iam.PolicyStatement({
            resources: ['*'],
            principals: [new iam.AnyPrincipal()],
            actions: ['es:*'],
          }),
        ],
      }),
      vpc: props.vpc,
    }).addInternalIngress();

    this.ecrImitari = new ecr.Repository(this, 'imitari', {
      repositoryName: 'imitari',
    });
  }
}
