import * as cdk from '@aws-cdk/core';
import * as route53 from '@aws-cdk/aws-route53';
import * as s3 from '@aws-cdk/aws-s3';
import * as sns from '@aws-cdk/aws-sns';
import * as snsSubscriptions from '@aws-cdk/aws-sns-subscriptions';
import * as iam from '@aws-cdk/aws-iam';
import * as ri from '@aws-cdk/region-info';
import { Config, BUKKIT_LEGACY_ACC_ID } from './config';

/**
 * Common resources among all or a majority of stacks, e.g. global secrets.
 */
export class CommonStack extends cdk.Stack {
  internalZone: route53.HostedZone;
  servicesZone: route53.HostedZone;
  cdnZone: route53.HostedZone;
  accessLogs: s3.Bucket;
  backupStorage: s3.Bucket;
  alertsTopic: sns.Topic;

  constructor(scope: cdk.Construct, config: Config) {
    super(scope, config.prefix + 'Common', { env: config.env });

    this.alertsTopic = new sns.Topic(this, 'AlertsTopic', {
      topicName: `${config.envName}Alerts`,
    });
    if (config.alertsEmail !== undefined) {
      this.alertsTopic.addSubscription(new snsSubscriptions.EmailSubscription(config.alertsEmail));
    }

    this.internalZone = new route53.HostedZone(this, 'StagingZone', {
      zoneName: 'overwolf.wtf',
    });

    this.servicesZone = new route53.HostedZone(this, 'ServicesZone', {
      zoneName: 'forgesvc.net',
    });

    this.cdnZone = new route53.HostedZone(this, 'CDNZone', {
      zoneName: 'forgecdn.net',
    });

    this.accessLogs = new s3.Bucket(this, 'AccessLogs', {
      blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
    });

    const account = ri.RegionInfo.get(this.region).elbv2Account;
    if (!account) {
      throw new Error(`Cannot enable access logging; don't know ELBv2 account for region ${this.region}`);
    }

    this.accessLogs.grantWrite(new iam.AccountPrincipal(account));
    this.accessLogs.policy?.document.addStatements(
      new iam.PolicyStatement({
        actions: ['s3:ListBucket*', 's3:GetBucketLocation'],
        principals: [new iam.AccountPrincipal(account)],
        resources: [this.accessLogs.bucketArn],
      })
    );

    this.backupStorage = new s3.Bucket(this, 'BackupStorage', {
      blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
    });

    // Allow access to dump backups from the legacy bukkit account
    this.backupStorage.grantReadWrite(new iam.AccountPrincipal(BUKKIT_LEGACY_ACC_ID));

    // This is required to deploy services with ecs-deploy that use secrets.
    new iam.Role(this, 'AdminDeployRole', {
      description: 'Used to deploy ecs-services with secrets',
      assumedBy: new iam.AccountRootPrincipal(),
      inlinePolicies: {
        blanket_secrets_and_ecs: new iam.PolicyDocument({
          statements: [
            new iam.PolicyStatement({
              actions: ['ecs:*', 'secretsmanager:Describe*', 'secretsmanager:Get*', 'secretsmanager:List*'],
              resources: ['*'],
            }),
          ],
        }),
      },
    });
  }
}
