import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as ecs from '@aws-cdk/aws-ecs';
import * as ecsPatterns from '@aws-cdk/aws-ecs-patterns';
import * as iam from '@aws-cdk/aws-iam';
import * as logs from '@aws-cdk/aws-logs';
import * as route53 from '@aws-cdk/aws-route53';
import * as targets from '@aws-cdk/aws-route53-targets';
import * as ec2 from '@aws-cdk/aws-ec2';
import { DlsDataStack } from './dls-data-stack';
import { Config } from './config';
import { CommonStack } from './common-stack';

const LEGACY_ELERIUM_ELASTICSEARCH_BACKUPS =
  'arn:aws:s3:::sql-scratch-space-default-useast1-79b0cfe9fbdf1386e0b9d32483224';

export interface DlsComputeProps {
  cluster: ecs.Cluster;
  dataStack: DlsDataStack;
  commonStack: CommonStack;
  cert: certificateManager.Certificate;
  vpc: ec2.Vpc;
}

export class DlsComputeStack extends cdk.Stack {
  constructor(scope: cdk.Construct, config: Config, props: DlsComputeProps) {
    super(scope, config.prefix + 'DlsCompute', { env: config.env });

    this.setupDownloadStats(config, props);
    this.setupDownloadEdge(config, props);
    this.setupDownloadStatsJobs(config, props);
    this.setupDownloadIngestES(config, props);
    this.setupDownloadIngestRedis(config, props);
    this.setupDownloadIngestDynamo(config, props);

    props.dataStack.redis.addIngressPeer(props.cluster.connections);

    this.setupIam(config, props);
  }

  private setupIam(config: Config, props: DlsComputeProps) {
    /* Iam role that allows ElasticSearch to talk to S3 (in this case from the legacy account, but can be adjusted for any S3) */
    const role = new iam.Role(this, 'ElasticSearchSnapshotRole', {
      assumedBy: new iam.ServicePrincipal('es.amazonaws.com'),
    });

    role.addToPolicy(
      new iam.PolicyStatement({
        resources: [LEGACY_ELERIUM_ELASTICSEARCH_BACKUPS],
        actions: ['s3:ListBucket'],
      })
    );

    role.addToPolicy(
      new iam.PolicyStatement({
        resources: [LEGACY_ELERIUM_ELASTICSEARCH_BACKUPS + '/Overwolf/*'],
        actions: ['s3:GetObject', 's3:PutObject', 's3:DeleteObject'],
      })
    );
  }

  private setupDownloadEdge(config: Config, props: DlsComputeProps) {
    const logGroup = new logs.LogGroup(this, 'DownloadEdgeContainerLogGroup', {
      logGroupName: 'DownloadEdge',
      retention: logs.RetentionDays.SIX_MONTHS,
    });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'EdgeTask');

    taskDefinition
      .addContainer('DownloadEdgeContainer', {
        image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrDownloadStatsEdge),
        logging: new ecs.AwsLogDriver({
          streamPrefix: 'Task',
          logGroup,
        }),
        memoryLimitMiB: 1024,
        environment: {
          'DownloadTracker:DisableFilenameEncoding': 'true',
          'DownloadTracker:ElasticSQSQueueAddress': props.dataStack.elasticQueue.queueUrl,
          'DownloadTracker:RedisSQSQueueAddress': props.dataStack.redisQueue.queueUrl,
          'DownloadTracker:DynamoSQSQueueAddress': props.dataStack.dynamoQueue.queueUrl,
          'DownloadTracker:RedirectHost': config.dlsRedirectHost,
          'DownloadTracker:SaveDatabaseInterval': '15',
        },
      })
      .addPortMappings({ containerPort: 80 });

    /* This service counts incoming download requests and forwards the user to the download */
    const service = new ecsPatterns.ApplicationLoadBalancedEc2Service(this, 'EdgeService', {
      cluster: props.cluster,
      taskDefinition,
      desiredCount: 4,
      certificate: props.cert,
      domainZone: props.commonStack.internalZone,
      domainName: 'edge-service.overwolf.wtf',
    });

    service.targetGroup.configureHealthCheck({
      path: '/api/health',
    });

    // DNS records
    new route53.ARecord(this, 'CdnZoneEdgeARecord', {
      recordName: 'edge.forgecdn.net',
      zone: props.commonStack.cdnZone,
      target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(service.loadBalancer)),
    });

    new route53.ARecord(this, 'CdnZoneFilesARecord', {
      recordName: 'files.forgecdn.net',
      zone: props.commonStack.cdnZone,
      target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(service.loadBalancer)),
    });

    // Assign permissions
    props.dataStack.elasticQueue.grantSendMessages(taskDefinition.taskRole);
    props.dataStack.redisQueue.grantSendMessages(taskDefinition.taskRole);
    props.dataStack.dynamoQueue.grantSendMessages(taskDefinition.taskRole);
    logGroup.grantWrite(taskDefinition.taskRole);
  }

  private setupDownloadStats(config: Config, props: DlsComputeProps) {
    const logGroup = new logs.LogGroup(this, 'DownloadStatsLogGroup', {
      logGroupName: 'DownloadStats',
      retention: logs.RetentionDays.SIX_MONTHS,
    });
    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'StatsTask');

    taskDefinition
      .addContainer('DownloadStatsContainer', {
        image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrDownloadStats),
        logging: new ecs.AwsLogDriver({
          streamPrefix: 'Task',
          logGroup,
        }),
        memoryLimitMiB: 1024,
        environment: {
          'DownloadsStats:ElasticSearchAddress': 'https://' + props.dataStack.elasticsearch.attrDomainEndpoint,
          'DownloadsStats:RedisAddress': props.dataStack.redis.attrPrimaryEndPointAddress,
          'DownloadsStats:DynamoTableNameProjects': props.dataStack.dynamoProjectTable.tableName,
          'DownloadsStats:DynamoTableNameProjectFiles': props.dataStack.dynamoProjectFilesTable.tableName,
        },
      })
      .addPortMappings({ containerPort: 80 });

    /* This service calculates BD stats for internal use */
    const service = new ecsPatterns.ApplicationLoadBalancedEc2Service(this, 'StatsService', {
      cluster: props.cluster,
      taskDefinition,
      desiredCount: 1,
      certificate: props.cert,
      domainName: 'download-stats',
      domainZone: props.commonStack.internalZone,
      publicLoadBalancer: false,
    });

    service.loadBalancer.connections.allowTo(
      ec2.Peer.ipv4(props.vpc.vpcCidrBlock),
      new ec2.Port({ protocol: ec2.Protocol.ALL, stringRepresentation: '443' })
    );

    service.targetGroup.configureHealthCheck({
      path: '/api/health',
    });

    // Permissions
    logGroup.grantWrite(taskDefinition.taskRole);
    props.dataStack.dynamoProjectTable.grantReadData(taskDefinition.taskRole);
    props.dataStack.dynamoProjectFilesTable.grantReadData(taskDefinition.taskRole);
  }

  private setupDownloadStatsJobs(config: Config, props: DlsComputeProps) {
    const logGroup = new logs.LogGroup(this, 'DownloadStatsJobsLogGroup', {
      logGroupName: 'DownloadStatsJobs',
      retention: logs.RetentionDays.SIX_MONTHS,
    });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'StatsJobsTask');

    taskDefinition
      .addContainer('DownloadStatsJobsContainer', {
        image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrDownloadStatsJobs),
        logging: new ecs.AwsLogDriver({
          streamPrefix: 'Task',
          logGroup,
        }),
        memoryLimitMiB: 1024,
        environment: {
          'DownloadsStatsJob:Env': config.envName.toLowerCase(),
          'DownloadsStatsJob:DynamoTableNameProjects': props.dataStack.dynamoProjectTable.tableName,
          'DownloadsStatsJob:DynamoTableNameProjectFiles': props.dataStack.dynamoProjectFilesTable.tableName,
          'DownloadsStatsJob:ElasticSearchAddress': 'https://' + props.dataStack.elasticsearch.attrDomainEndpoint,
        },
      })
      .addPortMappings({ containerPort: 80 });

    /* This service runs all the calculations against DLS in regards to popularity/points */
    const service = new ecsPatterns.ApplicationLoadBalancedEc2Service(this, 'StatsJobsService', {
      cluster: props.cluster,
      taskDefinition,
      desiredCount: 1,
    });

    service.targetGroup.configureHealthCheck({
      path: '/api/health',
    });

    // Permissions
    logGroup.grantWrite(taskDefinition.taskRole);
    props.dataStack.dynamoProjectTable.grantReadWriteData(taskDefinition.taskRole);
    props.dataStack.dynamoProjectFilesTable.grantReadWriteData(taskDefinition.taskRole);
  }

  private setupDownloadIngestES(config: Config, props: DlsComputeProps) {
    const logGroup = new logs.LogGroup(this, 'DownloadIngestElasticSearchLogGroup', {
      logGroupName: 'DownloadIngestElastic',
      retention: logs.RetentionDays.SIX_MONTHS,
    });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'IngestElastic');

    taskDefinition.addContainer('DownloadIngestElasticContainer', {
      image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrDownloadIngest),
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'Task',
        logGroup,
      }),
      memoryLimitMiB: 1174,
      environment: {
        'DownloadIngest:Env': config.envName.toLowerCase(),
        'DownloadIngest:ElasticSearchAddress': 'https://' + props.dataStack.elasticsearch.attrDomainEndpoint,
        'DownloadIngest:ElasticSQSQueueAddress': props.dataStack.elasticQueue.queueUrl,
        'DownloadIngest:SleepTimeout': '0', // Seconds to sleep between fetches. 0 = don't wait
        'DownloadIngest:Type': 'elastic',
        'DownloadIngest:EleriumFilesApiDomain': 'files-service.overwolf.wtf',
      },
    });

    /* This service ingests all download counts into elasticsearch for use with the stats services. */
    new ecs.Ec2Service(this, 'IngestElasticService', {
      cluster: props.cluster,
      taskDefinition,
      desiredCount: 6,
    });

    // Permissions
    logGroup.grantWrite(taskDefinition.taskRole);
    props.dataStack.elasticQueue.grantConsumeMessages(taskDefinition.taskRole);
  }

  private setupDownloadIngestRedis(config: Config, props: DlsComputeProps) {
    const logGroup = new logs.LogGroup(this, 'DownloadIngestRedisLogGroup', {
      logGroupName: 'DownloadIngestRedis',
      retention: logs.RetentionDays.SIX_MONTHS,
    });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'InjestRedis');

    taskDefinition.addContainer('DownloadIngestRedisContainer', {
      image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrDownloadIngest),
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'Task',
        logGroup,
      }),
      memoryLimitMiB: 764,
      environment: {
        'DownloadIngest:Env': config.envName.toLowerCase(),
        'DownloadIngest:RedisAddress':
          props.dataStack.redis.attrPrimaryEndPointAddress + ':' + props.dataStack.redis.attrPrimaryEndPointPort,
        'DownloadIngest:RedisSQSQueueAddress': props.dataStack.redisQueue.queueUrl,
        'DownloadIngest:SleepTimeout': '0', // Seconds to sleep between fetches. 0 = don't wait
        'DownloadIngest:Type': 'redis',
        'DownloadIngest:EleriumFilesApiDomain': 'files-service.overwolf.wtf',
      },
    });

    /* This service ingests all download counts into Redis, this keeps a tally of all downloads per file/project for instant access in Elerium */
    new ecs.Ec2Service(this, 'IngestRedisService', {
      cluster: props.cluster,
      taskDefinition,
      desiredCount: 4,
    });

    // Permissions
    logGroup.grantWrite(taskDefinition.taskRole);
    props.dataStack.redisQueue.grantConsumeMessages(taskDefinition.taskRole);
  }

  private setupDownloadIngestDynamo(config: Config, props: DlsComputeProps) {
    const logGroup = new logs.LogGroup(this, 'DownloadIngestDynamoLogGroup', {
      logGroupName: 'DownloadIngestDynamo',
      retention: logs.RetentionDays.SIX_MONTHS,
    });

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'IngestDynamo');

    taskDefinition.addContainer('DownloadIngestDynamoContainer', {
      image: ecs.ContainerImage.fromEcrRepository(props.dataStack.ecrDownloadIngest),
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'Task',
        logGroup,
      }),
      memoryLimitMiB: 764,
      environment: {
        'DownloadIngest:Env': config.envName.toLowerCase(),
        'DownloadIngest:DynamoSQSQueueAddress': props.dataStack.dynamoQueue.queueUrl,
        'DownloadIngest:SleepTimeout': '0', // Seconds to sleep between fetches. 0 = don't wait
        'DownloadIngest:Type': 'dynamo',
        'DownloadIngest:EleriumFilesApiDomain': 'files-service.overwolf.wtf',
      },
    });

    /* This service ingests all download counts into Dynamo as cold storage of all data. */
    new ecs.Ec2Service(this, 'IngestDynamoService', {
      cluster: props.cluster,
      taskDefinition,
      desiredCount: 4,
    });

    // Permissions
    logGroup.grantWrite(taskDefinition.taskRole);
    props.dataStack.dynamoQueue.grantConsumeMessages(taskDefinition.taskRole);
    props.dataStack.dynamoTable.grantWriteData(taskDefinition.taskRole);
  }
}
