import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as cloudfront from '@aws-cdk/aws-cloudfront';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as rds from '@aws-cdk/aws-rds';
import * as route53 from '@aws-cdk/aws-route53';
import * as targets from '@aws-cdk/aws-route53-targets';
import * as s3 from '@aws-cdk/aws-s3';
import * as sm from '@aws-cdk/aws-secretsmanager';
import * as sns from '@aws-cdk/aws-sns';
import * as snsSubscriptions from '@aws-cdk/aws-sns-subscriptions';
import * as ses from '@aws-cdk/aws-ses';
import { Config } from './config';
import { CommonStack } from './common-stack';
import { ElasticCache } from './resources/elastic-cache';

export interface EleriumDataStackProps {
  vpc: ec2.Vpc;
  cert: certificateManager.Certificate;
  commonStack: CommonStack;
  backupsS3: s3.Bucket;
}

export class EleriumDataStack extends cdk.Stack {
  // Stack Constants
  // Note: When using this URL ensure that the stack dependency chain is maintained.
  // newStack.requires(eleriumDataStack);
  ELERIUM_ASSETS_CDN_URL = 'media-cdn.overwolf.wtf';

  // Stack Assets
  db: rds.DatabaseInstance;
  assetsBucket: s3.Bucket;
  globalSecretProvider: sm.Secret;
  databaseSecret: sm.Secret;
  eleriumVersions: s3.Bucket;
  smtpSecret: sm.Secret;
  textProviderSecret: sm.Secret;
  recaptchaSecret: sm.Secret;
  twitchClientSecrets: sm.Secret;
  dataManagerInvalidationsSns: sns.Topic;
  redis: ElasticCache;

  constructor(scope: cdk.Construct, config: Config, props: EleriumDataStackProps) {
    super(scope, config.prefix + 'EleriumData', { env: config.env });

    const rdsIamRole = new iam.Role(this, 'EleriumOptionsGroupServiceRole', {
      assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
      inlinePolicies: {
        native_backups: new iam.PolicyDocument({
          statements: [
            new iam.PolicyStatement({
              actions: ['s3:ListBucket', 's3:GetBucketLocation'],
              resources: [props.backupsS3.bucketArn],
            }),
            new iam.PolicyStatement({
              actions: [
                's3:GetObjectMetaData',
                's3:GetObject',
                's3:PutObject',
                's3:ListMultipartUploadParts',
                's3:AbortMultipartUpload',
              ],
              resources: [props.backupsS3.arnForObjects('*')],
            }),
          ],
        }),
      },
    });

    const rdsOptionGroup = new rds.OptionGroup(this, 'EleriumOptionsGroup', {
      engine: rds.DatabaseInstanceEngine.sqlServerEe({ version: rds.SqlServerEngineVersion.VER_13_00_5598_27_V1 }),
      configurations: [{ name: 'SQLSERVER_BACKUP_RESTORE', settings: { IAM_ROLE_ARN: rdsIamRole.roleArn } }],
    });

    this.db = new rds.DatabaseInstance(this, 'Elerium', {
      allocatedStorage: 1000,
      engine: rds.DatabaseInstanceEngine.sqlServerEe({ version: rds.SqlServerEngineVersion.VER_13_00_5598_27_V1 }),
      instanceIdentifier: 'elerium-prod',
      instanceType: ec2.InstanceType.of(ec2.InstanceClass.R5, ec2.InstanceSize.XLARGE16),
      iops: 10000,
      licenseModel: rds.LicenseModel.LICENSE_INCLUDED,
      masterUsername: 'administrator',
      multiAz: true,
      storageType: rds.StorageType.IO1,
      vpc: props.vpc,
      enablePerformanceInsights: true,
      optionGroup: rdsOptionGroup,
    });

    // Allow connections from VPN
    this.db.connections.allowDefaultPortFrom(ec2.Peer.ipv4(props.vpc.vpcCidrBlock));

    // CloudFront origin access identity
    const originAccessIdentity = new cloudfront.OriginAccessIdentity(this, 'OriginAccessIdentity');

    // Assets
    this.assetsBucket = new s3.Bucket(this, 'AssetsBucket', {
      blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
      cors: [
        {
          allowedHeaders: ['*'],
          allowedMethods: [
            s3.HttpMethods.DELETE,
            s3.HttpMethods.GET,
            s3.HttpMethods.HEAD,
            s3.HttpMethods.POST,
            s3.HttpMethods.PUT,
          ],
          allowedOrigins: [
            'BUKKIT.ORG',
            'CURSEFORGE.COM',
            'CURSEFORGE.NET',
            'FORGECDN.NET',
            'FORGESVC.NET',
            'SC2MAPSTER.COM',
            'WOWACE.COM',
          ],
          maxAge: 3000,
        },
      ],
    });
    this.assetsBucket.grantRead(originAccessIdentity);

    // Assets distribution
    const assetsDist = new cloudfront.CloudFrontWebDistribution(this, 'AssetsDist', {
      originConfigs: [
        {
          s3OriginSource: {
            s3BucketSource: this.assetsBucket,
            originAccessIdentity,
          },
          behaviors: [{ isDefaultBehavior: true }],
        },
      ],
      viewerCertificate: cloudfront.ViewerCertificate.fromAcmCertificate(props.cert, {
        aliases: ['media.forgecdn.net', 'media-cdn.overwolf.wtf'],
      }),
      priceClass: cloudfront.PriceClass.PRICE_CLASS_ALL,
    });

    // Assets DNS records
    new route53.ARecord(this, 'AssetsDistARecord', {
      recordName: 'media-cdn.overwolf.wtf',
      zone: props.commonStack.internalZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(assetsDist)),
    });

    new route53.AaaaRecord(this, 'AssetsDistAaaaRecord', {
      recordName: 'media-cdn.overwolf.wtf',
      zone: props.commonStack.internalZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(assetsDist)),
    });

    new route53.ARecord(this, 'AssetsDistCdnZoneARecord', {
      recordName: 'media.forgecdn.net',
      zone: props.commonStack.cdnZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(assetsDist)),
    });

    new route53.AaaaRecord(this, 'AssetsDistCdnZoneAaaaRecord', {
      recordName: 'media.forgecdn.net',
      zone: props.commonStack.cdnZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(assetsDist)),
    });

    // The deploying entity needs permissions on this bucket.
    this.eleriumVersions = new s3.Bucket(this, 'BeanstalkVersionsBucket');

    // These are json secret that needs to be filled in manually.
    this.globalSecretProvider = new sm.Secret(this, 'EleriumGlobalSecretProvider', {
      description: 'Cobalt global provider api key and crypto key',
      secretName: 'Elerium/' + config.envName + '/Global-Provider-Secret',
    });

    this.twitchClientSecrets = new sm.Secret(this, 'EleriumTwitchClientSecrets', {
      description: 'Twitch Client secrets for each of the domain running on Elerium',
      secretName: 'Elerium/' + config.envName + '/Twitch-Client-Secrets',
    });

    this.recaptchaSecret = new sm.Secret(this, 'EleriumRecaptchaSecret', {
      description: 'Secret key for Recaptcha',
      secretName: 'Elerium/Recaptcha-Secret',
    });

    this.textProviderSecret = new sm.Secret(this, 'EleriumTextProviderSecret', {
      description: 'Secret token for text message provider',
      secretName: 'Global/Text-Message-Provider-Secret',
    });

    this.smtpSecret = new sm.Secret(this, 'EleriumSmtp', {
      description: 'Aws Smtp Creds',
      secretName: 'Global/Smtp',
    });

    // Normally this would be a DB secret but we need to control the secret.
    this.databaseSecret = new sm.Secret(this, 'EleriumApplicationDBSecret', {
      description: 'Production Creds for Elerium RDS Instance for Elerium Application',
      secretName: 'Elerium/' + config.envName + '/RDS',
    });

    this.dataManagerInvalidationsSns = new sns.Topic(this, 'EleriumDataManagerCacheInvalidations', {
      displayName: "Forwards Elerium's cache invalidations to DataManager's SQS Queue",
    });

    // SES notifications secret
    const sesNotificationSecret = new sm.Secret(this, 'EmailNotificationsSecret', {
      secretName: `Elerium/${config.envName}/EmailNotifications`,
    });

    // SES notifications topic.
    const sesNotificationsTopic = new sns.Topic(this, 'EmailNotificationsTopic');
    sesNotificationsTopic.addSubscription(
      new snsSubscriptions.UrlSubscription(
        `${config.sesNotificationsUrl}?v=${config.sesNotificationsSecretVersion}&token=${sesNotificationSecret.secretValue}`,
        {
          protocol: sns.SubscriptionProtocol.HTTPS,
        }
      )
    );

    // Redis for CrashLogs
    this.redis = new ElasticCache(this, config.prefix + `EleriumCrashLogs RedisCluster`, {
      cacheNodeType: config.addonServiceElasticServerType,
      replicasPerNodeGroup: config.addonServiceElasticServerCount,
      replicationGroupDescription: 'EleriumCrashLogs Redis Cluster',
      vpc: props.vpc,
    });

    // SES notifications config set
    // Important:
    //   CloudFormation doesn't support Config Set SNS destinations, so we need to manually create the destination for the above topic in the console.
    //   It should receive the following events: Send, Reject, Delivery, Open, Click, Bounce, Complaint.
    new ses.CfnConfigurationSet(this, 'EmailNotificationsConfigSet');
  }
}
