import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as s3 from '@aws-cdk/aws-s3';
import * as cf from '@aws-cdk/aws-cloudfront';
import { Config } from './config';
import { CommonStack } from './common-stack';

interface GtaGamingStackProps {
  commonStack: CommonStack;
}

/**
 * GTA Gaming Stack
 */
export class GtaGamingStack extends cdk.Stack {
  constructor(scope: cdk.Construct, config: Config, props: GtaGamingStackProps) {
    super(scope, config.prefix + 'GtaGaming', { env: config.env });

    // CloudFront origin access identity
    const originAccessIdentity = new cf.OriginAccessIdentity(this, 'OriginAccessIdentity');

    // S3 bucket
    const bucket = new s3.Bucket(this, 'Bucket', {
      blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
    });
    bucket.grantRead(originAccessIdentity);

    // Cert
    const cert = new certificateManager.Certificate(this, 'Certificate', {
      validationMethod: certificateManager.ValidationMethod.DNS,
      domainName: 'gtagaming.com',
      subjectAlternativeNames: ['*.gtagaming.com', 'gta4-mods.com', '*.gta4-mods.com'],
    });

    // CloudFront distribution
    new cf.CloudFrontWebDistribution(this, 'Distribution', {
      originConfigs: [
        {
          s3OriginSource: {
            s3BucketSource: bucket,
            originAccessIdentity,
          },
          behaviors: [{ isDefaultBehavior: true }],
        },
      ],
      viewerCertificate: cf.ViewerCertificate.fromAcmCertificate(cert, {
        aliases: ['shadow.gtagaming.com', 'www.gtagaming.com', 'gtagaming.com', 'www.gta4-mods.com', 'gta4-mods.com'],
      }),
      priceClass: cf.PriceClass.PRICE_CLASS_ALL,
    });
  }
}
