import * as cdk from '@aws-cdk/core';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as rds from '@aws-cdk/aws-rds';
import * as ecr from '@aws-cdk/aws-ecr';
import * as sm from '@aws-cdk/aws-secretsmanager';
import { Config } from './config';
import * as s3 from '@aws-cdk/aws-s3';

export interface HyruleDataStackProps {
  vpc: ec2.Vpc;
}

export class HyruleDataStack extends cdk.Stack {
  databaseCluster: rds.DatabaseCluster;
  ecrHyrule: ecr.Repository;
  googleSecret: sm.Secret;
  eleriumApiSecret: sm.Secret;
  s3Bucket: s3.Bucket;

  constructor(scope: cdk.Construct, config: Config, props: HyruleDataStackProps) {
    super(scope, config.prefix + 'HyruleData', { env: config.env });

    this.ecrHyrule = new ecr.Repository(this, 'hyrule', {
      repositoryName: 'hyrule',
    });

    // S3 bucket
    this.s3Bucket = new s3.Bucket(this, 'HyruleBucket', {
      blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
    });

    this.databaseCluster = new rds.DatabaseCluster(this, 'DatabaseCluster', {
      engine: rds.DatabaseClusterEngine.auroraMysql({
        version: rds.AuroraMysqlEngineVersion.VER_2_08_1,
      }),
      instanceProps: {
        vpc: props.vpc,
        instanceType: ec2.InstanceType.of(ec2.InstanceClass.R4, ec2.InstanceSize.LARGE),
      },
      masterUser: { username: 'administrator' },
    });

    this.googleSecret = new sm.Secret(this, 'GoogleSecret', {
      description: 'Secret token for google auth',
      secretName: 'Hyrule/GoogleAuth',
    });

    this.eleriumApiSecret = new sm.Secret(this, 'EleriumApiSecret', {
      description: 'Secret token for google auth',
      secretName: 'Hyrule/EleriumApi',
    });
  }
}
