import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as iam from '@aws-cdk/aws-iam';
import * as elasticbeanstalk from '@aws-cdk/aws-elasticbeanstalk';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as route53 from '@aws-cdk/aws-route53';
import { Config } from './config';
import { CommonStack } from './common-stack';
import { EleriumDataStack } from './elerium-data-stack';
import { Helper } from './resources/helper';
import { LegacyAcsDataStack } from './legacy-acs-data-stack';
import { ElasticBeanstalkTarget } from './resources/elastic-beanstalk-target';

export interface LegacyAcsComputeProps {
  dataStack: LegacyAcsDataStack;
  eleriumDataStack: EleriumDataStack;
  commonStack: CommonStack;
  cert: certificateManager.Certificate;
  vpc: ec2.Vpc;
}

export class LegacyAcsComputeStack extends cdk.Stack {
  constructor(scope: cdk.Construct, config: Config, props: LegacyAcsComputeProps) {
    super(scope, config.prefix + 'LegacyAcsCompute', { env: config.env });

    this.setupModloaders(config, props);
    this.setupServerMods(config, props);
  }

  private setupModloaders(config: Config, props: LegacyAcsComputeProps) {
    // Production
    const modloadersIamRole = new iam.Role(this, 'ModloadersServiceRole', {
      assumedBy: new iam.ServicePrincipal('elasticbeanstalk.amazonaws.com'),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSElasticBeanstalkService'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSElasticBeanstalkEnhancedHealth'),
      ],
    });

    const modloadersIamInstanceRole = new iam.Role(this, 'ModloadersEnvInstanceRole', {
      assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkWebTier'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkMulticontainerDocker'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkWorkerTier'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
      ],
      inlinePolicies: {
        // These are required for the beanstalk extension scripts to function
        beanstalk_extensions: new iam.PolicyDocument({
          statements: [
            new iam.PolicyStatement({
              actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:GetLogEvents',
                'logs:PutLogEvents',
                'logs:DescribeLogGroups',
                'logs:DescribeLogStreams',
                'logs:PutRetentionPolicy',
              ],
              resources: ['*'],
            }),
          ],
        }),
        // I have no idea what this is for.
        sqs: new iam.PolicyDocument({
          statements: [
            new iam.PolicyStatement({
              actions: [
                'sqs:DeleteMessage',
                'sqs:GetQueueUrl',
                'sqs:ListQueues',
                'sqs:ChangeMessageVisibility',
                'sqs:SendMessageBatch',
                'sqs:SendMessage',
                'sqs:GetQueueAttributes',
                'sqs:ListQueueTags',
                'sqs:ListDeadLetterSourceQueues',
                'sqs:DeleteMessageBatch',
                'sqs:PurgeQueue',
                'sqs:DeleteQueue',
                'sqs:CreateQueue',
                'sqs:ChangeMessageVisibilityBatch',
                'sqs:SetQueueAttributes',
              ],
              resources: ['*'],
            }),
          ],
        }),
        secret_manager: new iam.PolicyDocument({
          statements: [
            new iam.PolicyStatement({
              actions: ['secretsmanager:Describe*', 'secretsmanager:Get*', 'secretsmanager:List*'],
              resources: [
                props.eleriumDataStack.databaseSecret.secretArn,
                props.dataStack.modLoadersDatabaseSecret.secretArn,
              ],
            }),
          ],
        }),
      },
    });

    const modloadersIamInstanceProfile = new iam.CfnInstanceProfile(this, 'ModloaderEnvInstanceRoleProfile', {
      roles: [modloadersIamInstanceRole.roleName],
    });

    modloadersIamInstanceProfile.node.addDependency(modloadersIamInstanceRole);

    const modloadersEbSecurityGroup = new ec2.SecurityGroup(this, 'ModloaderBeanstalkSecurityGroup', {
      vpc: props.vpc,
    });

    const modloadersLbSecurityGroup = new ec2.SecurityGroup(this, 'ModloaderLoadBalancerSecurityGroup', {
      vpc: props.vpc,
    });

    const app = new elasticbeanstalk.CfnApplication(this, 'ModLoadersBeanstalkApp', {
      applicationName: 'modloaders',
      resourceLifecycleConfig: {
        serviceRole: modloadersIamRole.roleArn,
      },
    });

    app.node.addDependency(modloadersIamRole);

    const env = new elasticbeanstalk.CfnEnvironment(this, 'ModLoadersBeanstalkEnv', {
      applicationName: app.applicationName!,
      solutionStackName: '64bit Windows Server Core 2016 v2.5.8 running IIS 10.0',
      optionSettings: [
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'InstanceType',
          value: 'c5.xlarge',
        },
        {
          namespace: 'aws:autoscaling:asg',
          optionName: 'MinSize',
          value: config.legacyACSBeanstalkMinCount.toString(),
        },
        {
          namespace: 'aws:autoscaling:asg',
          optionName: 'MaxSize',
          value: config.legacyACSBeanstalkMaxCount.toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:cloudwatch:logs',
          optionName: 'StreamLogs',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:cloudwatch:logs',
          optionName: 'RetentionInDays',
          value: '365',
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'AccessLogsS3Bucket',
          value: props.commonStack.accessLogs.bucketName,
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'AccessLogsS3Enabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment',
          optionName: 'LoadBalancerType',
          value: 'application',
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'VPCId',
          value: props.vpc.vpcId,
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'Subnets',
          value: props.vpc.publicSubnets.map((s) => s.subnetId).join(','),
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'ELBSubnets',
          value: props.vpc.publicSubnets.map((s) => s.subnetId).join(','),
        },
        {
          namespace: 'aws:elasticbeanstalk:environment',
          optionName: 'ServiceRole',
          value: modloadersIamRole.roleName,
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'IamInstanceProfile',
          value: modloadersIamInstanceProfile.ref, // Ref on an instance profile is the name
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'DeploymentPolicy',
          value: 'Rolling',
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'BatchSizeType',
          value: 'Fixed',
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'BatchSize',
          value: '1',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'RollingUpdateEnabled',
          value: 'true',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'MaxBatchSize',
          value: '1',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'MinInstancesInService',
          value: config.legacyACSBeanstalkMinCount.toString(),
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'RollingUpdateType',
          value: 'Health',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'DeregistrationDelay',
          value: '10',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckInterval',
          value: '5',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthyThresholdCount',
          value: config.legacyACSBeanstalkMaxCount.toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'UnhealthyThresholdCount',
          value: config.legacyACSBeanstalkMinCount.toString(),
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'MonitoringInterval',
          value: '1 Minute',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'RootVolumeSize',
          value: '50',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'RootVolumeType',
          value: 'gp2',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'SecurityGroups',
          value: modloadersEbSecurityGroup.securityGroupName,
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'SecurityGroups',
          value: modloadersLbSecurityGroup.securityGroupName,
        },
        {
          namespace: 'aws:elbv2:listener:443',
          optionName: 'Protocol',
          value: 'HTTPS',
        },
        {
          namespace: 'aws:elbv2:listener:443',
          optionName: 'SSLCertificateArns',
          value: props.cert.certificateArn,
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'AssociatePublicIpAddress',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckPath',
          value: '/health-check',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'MatcherHTTPCode',
          value: '200',
        },
        {
          namespace: 'aws:elasticbeanstalk:hostmanager',
          optionName: 'LogPublicationControl',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'IgnoreHealthCheck',
          value: 'false',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'DeregistrationDelay',
          value: '10',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'StickinessEnabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckTimeout',
          value: '2',
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'IdleTimeout',
          value: '600',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions',
          optionName: 'ManagedActionsEnabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions',
          optionName: 'PreferredStartTime',
          value: 'Thu:10:10',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions:platformupdate',
          optionName: 'UpdateLevel',
          value: 'minor',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions:platformupdate',
          optionName: 'InstanceRefreshEnabled',
          value: 'false',
        },
        {
          namespace: 'aws:elasticbeanstalk:healthreporting:system',
          optionName: 'SystemType',
          value: 'enhanced',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'MODLOADERS_BASE_URL',
          value: 'https://modloaders-cdn.overwolf.wtf', // TODO replace with a dynamic url?
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'MODLOADERS_S3_BUCKET',
          value: props.dataStack.modloadersAssetsBuckets.bucketName,
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_DEFAULT_USERNAME',
          value: props.eleriumDataStack.db.secret!.secretValueFromJson('username').toString(), // TODO temporary use admin creds
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_DEFAULT_PASSWORD',
          value: props.eleriumDataStack.db.secret!.secretValueFromJson('password').toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_DEFAULT_HOSTNAME',
          value: props.eleriumDataStack.db.secret!.secretValueFromJson('host').toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'ELERIUM_API_KEY',
          value: props.dataStack.eleriumApiKey.secretValue.toString(),
        },
      ],
    });

    env.addDependsOn(app);
    env.node.addDependency(modloadersIamRole);
    env.node.addDependency(modloadersIamInstanceRole);

    Helper.addDefaultPortIngress(props.eleriumDataStack.db.connections, modloadersEbSecurityGroup);
    props.dataStack.modloadersAssetsBuckets.grantReadWrite(modloadersIamInstanceRole);
  }

  private setupServerMods(config: Config, props: LegacyAcsComputeProps) {
    // Production
    const serverModsIamRole = new iam.Role(this, 'ServerModsServiceRole', {
      assumedBy: new iam.ServicePrincipal('elasticbeanstalk.amazonaws.com'),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSElasticBeanstalkService'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSElasticBeanstalkEnhancedHealth'),
      ],
    });

    const serverModsIamInstanceRole = new iam.Role(this, 'SeverModsEnvInstanceRole', {
      assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkWebTier'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkMulticontainerDocker'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AWSElasticBeanstalkWorkerTier'),
        iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
      ],
      inlinePolicies: {
        secret_manager: new iam.PolicyDocument({
          statements: [
            new iam.PolicyStatement({
              actions: ['secretsmanager:Describe*', 'secretsmanager:Get*', 'secretsmanager:List*'],
              resources: [
                props.eleriumDataStack.databaseSecret.secretArn,
                props.dataStack.serverModsDatabaseSecret.secretArn,
              ],
            }),
          ],
        }),
      },
    });

    const serverModsIamInstanceProfile = new iam.CfnInstanceProfile(this, 'ServerModsEnvInstanceRoleProfile', {
      roles: [serverModsIamInstanceRole.roleName],
    });

    serverModsIamInstanceProfile.node.addDependency(serverModsIamInstanceRole);

    const serverModsEbSecurityGroup = new ec2.SecurityGroup(this, 'ServerModsBeanstalkSecurityGroup', {
      vpc: props.vpc,
    });

    const serverModsLbSecurityGroup = new ec2.SecurityGroup(this, 'ServerModsLoadBalancerSecurityGroup', {
      vpc: props.vpc,
    });

    const app = new elasticbeanstalk.CfnApplication(this, 'ServerModsBeanstalkApp', {
      applicationName: 'servermods',
      resourceLifecycleConfig: {
        serviceRole: serverModsIamRole.roleArn,
      },
    });

    app.node.addDependency(serverModsIamRole);

    const env = new elasticbeanstalk.CfnEnvironment(this, 'ServerModsBeanstalkEnv', {
      applicationName: app.applicationName!,
      solutionStackName: '64bit Windows Server Core 2016 v2.5.8 running IIS 10.0',
      optionSettings: [
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'InstanceType',
          value: 'c5.xlarge',
        },
        {
          namespace: 'aws:autoscaling:asg',
          optionName: 'MinSize',
          value: config.legacyACSBeanstalkMinCount.toString(),
        },
        {
          namespace: 'aws:autoscaling:asg',
          optionName: 'MaxSize',
          value: config.legacyACSBeanstalkMaxCount.toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:cloudwatch:logs',
          optionName: 'StreamLogs',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:cloudwatch:logs',
          optionName: 'RetentionInDays',
          value: '365',
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'AccessLogsS3Bucket',
          value: props.commonStack.accessLogs.bucketName,
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'AccessLogsS3Enabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment',
          optionName: 'LoadBalancerType',
          value: 'application',
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'VPCId',
          value: props.vpc.vpcId,
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'Subnets',
          value: props.vpc.publicSubnets.map((s) => s.subnetId).join(','),
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'ELBSubnets',
          value: props.vpc.publicSubnets.map((s) => s.subnetId).join(','),
        },
        {
          namespace: 'aws:elasticbeanstalk:environment',
          optionName: 'ServiceRole',
          value: serverModsIamRole.roleName,
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'IamInstanceProfile',
          value: serverModsIamInstanceProfile.ref, // Ref on an instance profile is the name
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'DeploymentPolicy',
          value: 'Rolling',
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'BatchSizeType',
          value: 'Fixed',
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'BatchSize',
          value: '1',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'RollingUpdateEnabled',
          value: 'true',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'MaxBatchSize',
          value: '1',
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'MinInstancesInService',
          value: config.legacyACSBeanstalkMinCount.toString(),
        },
        {
          namespace: 'aws:autoscaling:updatepolicy:rollingupdate',
          optionName: 'RollingUpdateType',
          value: 'Health',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'DeregistrationDelay',
          value: '10',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckInterval',
          value: '5',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthyThresholdCount',
          value: config.legacyACSBeanstalkMaxCount.toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'UnhealthyThresholdCount',
          value: config.legacyACSBeanstalkMinCount.toString(),
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'MonitoringInterval',
          value: '1 Minute',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'RootVolumeSize',
          value: '50',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'RootVolumeType',
          value: 'gp2',
        },
        {
          namespace: 'aws:autoscaling:launchconfiguration',
          optionName: 'SecurityGroups',
          value: serverModsEbSecurityGroup.securityGroupName,
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'SecurityGroups',
          value: serverModsLbSecurityGroup.securityGroupName,
        },
        {
          namespace: 'aws:elbv2:listener:443',
          optionName: 'Protocol',
          value: 'HTTPS',
        },
        {
          namespace: 'aws:elbv2:listener:443',
          optionName: 'SSLCertificateArns',
          value: props.cert.certificateArn,
        },
        {
          namespace: 'aws:ec2:vpc',
          optionName: 'AssociatePublicIpAddress',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckPath',
          value: '/health-check',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'MatcherHTTPCode',
          value: '200',
        },
        {
          namespace: 'aws:elasticbeanstalk:hostmanager',
          optionName: 'LogPublicationControl',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:command',
          optionName: 'IgnoreHealthCheck',
          value: 'false',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'DeregistrationDelay',
          value: '10',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'StickinessEnabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:environment:process:default',
          optionName: 'HealthCheckTimeout',
          value: '2',
        },
        {
          namespace: 'aws:elbv2:loadbalancer',
          optionName: 'IdleTimeout',
          value: '600',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions',
          optionName: 'ManagedActionsEnabled',
          value: 'true',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions',
          optionName: 'PreferredStartTime',
          value: 'Thu:10:10',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions:platformupdate',
          optionName: 'UpdateLevel',
          value: 'minor',
        },
        {
          namespace: 'aws:elasticbeanstalk:managedactions:platformupdate',
          optionName: 'InstanceRefreshEnabled',
          value: 'false',
        },
        {
          namespace: 'aws:elasticbeanstalk:healthreporting:system',
          optionName: 'SystemType',
          value: 'enhanced',
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_DEFAULT_USERNAME',
          value: props.dataStack.serverModsDatabaseSecret.secretValueFromJson('username').toString(), // TODO temporary use admin creds
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_DEFAULT_PASSWORD',
          value: props.dataStack.serverModsDatabaseSecret.secretValueFromJson('password').toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'RDS_DEFAULT_HOSTNAME',
          value: props.eleriumDataStack.db.secret!.secretValueFromJson('host').toString(),
        },
        {
          namespace: 'aws:elasticbeanstalk:application:environment',
          optionName: 'ASPNET_ENV',
          value: config.envName,
        },
      ],
    });

    env.addDependsOn(app);
    env.node.addDependency(serverModsIamRole);
    env.node.addDependency(serverModsIamInstanceRole);

    Helper.addDefaultPortIngress(props.eleriumDataStack.db.connections, serverModsEbSecurityGroup);

    new route53.ARecord(this, 'ServerModsRecord', {
      target: route53.RecordTarget.fromAlias(new ElasticBeanstalkTarget(env)),
      zone: props.commonStack.servicesZone,
      recordName: 'servermods',
    });
  }
}
