import * as cdk from '@aws-cdk/core';
import * as certificateManager from '@aws-cdk/aws-certificatemanager';
import * as cloudfront from '@aws-cdk/aws-cloudfront';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as route53 from '@aws-cdk/aws-route53';
import * as targets from '@aws-cdk/aws-route53-targets';
import * as s3 from '@aws-cdk/aws-s3';
import * as sm from '@aws-cdk/aws-secretsmanager';
import * as rds from '@aws-cdk/aws-rds';
import { Config } from './config';
import { EleriumDataStack } from './elerium-data-stack';
import { CommonStack } from './common-stack';

export interface LegacyAcsDataStackProps {
  vpc: ec2.Vpc;
  cert: certificateManager.Certificate;
  commonStack: CommonStack;
  eleriumDataStack: EleriumDataStack;
}

export class LegacyAcsDataStack extends cdk.Stack {
  modloadersAssetsBuckets: s3.Bucket;
  eleriumApiKey: sm.Secret;
  serverModsDatabaseSecret: rds.DatabaseSecret;
  modLoadersDatabaseSecret: rds.DatabaseSecret;

  constructor(scope: cdk.Construct, config: Config, props: LegacyAcsDataStackProps) {
    super(scope, config.prefix + 'LegacyAcsData', { env: config.env });

    // CloudFront origin access identity
    const originAccessIdentity = new cloudfront.OriginAccessIdentity(this, 'OriginAccessIdentity');

    // Assets
    this.modloadersAssetsBuckets = new s3.Bucket(this, 'ModloadersAssetsBucket', {
      blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
    });
    this.modloadersAssetsBuckets.grantRead(originAccessIdentity);

    // Assets distribution
    const ModloadersAssetsDist = new cloudfront.CloudFrontWebDistribution(this, 'ModloadersAssetsDist', {
      originConfigs: [
        {
          s3OriginSource: {
            s3BucketSource: this.modloadersAssetsBuckets,
            originAccessIdentity,
          },
          behaviors: [{ isDefaultBehavior: true }],
        },
      ],
      viewerCertificate: cloudfront.ViewerCertificate.fromAcmCertificate(props.cert, {
        aliases: ['modloaders-cdn.overwolf.wtf', 'modloaders.forgecdn.net'],
      }),
      priceClass: cloudfront.PriceClass.PRICE_CLASS_ALL,
    });

    // Assets DNS records
    new route53.ARecord(this, 'ModloadersAssetsDistARecord', {
      recordName: 'modloaders-cdn.overwolf.wtf',
      zone: props.commonStack.internalZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(ModloadersAssetsDist)),
    });

    new route53.AaaaRecord(this, 'ModloadersAssetsDistAaaaRecord', {
      recordName: 'modloaders-cdn.overwolf.wtf',
      zone: props.commonStack.internalZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(ModloadersAssetsDist)),
    });

    new route53.ARecord(this, 'ModloadersAssetsDistCdnZoneARecord', {
      recordName: 'modloaders.forgecdn.net',
      zone: props.commonStack.cdnZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(ModloadersAssetsDist)),
    });

    new route53.AaaaRecord(this, 'ModloadersAssetsDistCdnZoneAaaaRecord', {
      recordName: 'modloaders.forgecdn.net',
      zone: props.commonStack.cdnZone,
      target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(ModloadersAssetsDist)),
    });

    // Keep it secret, keep it safe
    this.eleriumApiKey = new sm.Secret(this, 'ModLoadersEleriumApiKey', {
      description: 'API Key that allows ModLoaders to talk to Elerium',
    });

    this.serverModsDatabaseSecret = new rds.DatabaseSecret(this, 'ServerModsDatabaseSecret', {
      username: 'server_mods',
      masterSecret: props.eleriumDataStack.db.secret,
    });

    this.modLoadersDatabaseSecret = new rds.DatabaseSecret(this, 'ModLoadersDatabaseSecret', {
      username: 'mod_loaders',
      masterSecret: props.eleriumDataStack.db.secret,
    });
  }
}
