import * as ec2 from '@aws-cdk/aws-ec2';
import * as ec from '@aws-cdk/aws-elasticache';
import * as route53 from '@aws-cdk/aws-route53';
import * as cdk from '@aws-cdk/core';
import { ElasticCacheResourceProps } from './elastic-cache-resource-props';
import { IpWhitelistProps } from './ip-whitelist-props';

export class ElasticCache extends ec.CfnReplicationGroup {
  private securityGroup?: ec2.SecurityGroup;
  private vpc: ec2.Vpc;

  constructor(parent: cdk.Construct, name: string, props: ElasticCacheResourceProps) {
    super(parent, name, {
      autoMinorVersionUpgrade: true,
      automaticFailoverEnabled: true,
      cacheNodeType: 'cache.r4.large',
      engine: 'redis',
      numNodeGroups: 1,
      port: 6379, // Set for reference in later areas
      replicasPerNodeGroup: 2,
      ...(props as ec.CfnReplicationGroupProps),
    });

    this.cfnOptions.updatePolicy = {
      useOnlineResharding: true,
    };

    this.vpc = props.vpc;

    if (props.cacheSubnetGroupName === undefined) {
      this.setupSubnetGroup();
    }
  }

  public addIngressIp(ip: string) {
    this.createSecurityGroup();

    this.securityGroup!.addIngressRule(ec2.Peer.ipv4(ip), ec2.Port.tcp(this.port!));

    return this;
  }

  public addIngressIps(ips: IpWhitelistProps[]) {
    this.createSecurityGroup();

    for (const ip of ips) {
      this.securityGroup!.addIngressRule(ec2.Peer.ipv4(ip.Ip), ec2.Port.tcp(this.port!), ip.Description);
    }

    return this;
  }

  public addIngressSg(securityGroup: ec2.SecurityGroup) {
    this.createSecurityGroup();

    this.securityGroup!.addIngressRule(securityGroup, ec2.Port.tcp(this.port!));

    return this;
  }

  public addIngressPeer(peer: ec2.Connections, remote?: boolean) {
    this.createSecurityGroup();

    for (const sg of peer.securityGroups) {
      this.securityGroup!.addIngressRule(sg, ec2.Port.tcp(this.port!), undefined, remote);
    }

    return this;
  }

  public addAlias(zone: route53.IHostedZone, name: string, recordName: string, ttl?: cdk.Duration) {
    new route53.CnameRecord(this, name + 'ElasticCacheAlias', {
      domainName: this.attrPrimaryEndPointAddress,
      zone,
      recordName,
      ttl,
    });

    return this;
  }

  private setupSubnetGroup() {
    const subnetGroup = new ec.CfnSubnetGroup(this, 'PublicSubnets', {
      description: `Subnet group for ${this.node.id}`,
      subnetIds: this.vpc.publicSubnets.map((subnet) => subnet.subnetId),
    });

    this.cacheSubnetGroupName = subnetGroup.ref;
  }

  private createSecurityGroup() {
    if (this.securityGroup === undefined) {
      this.securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
        description: `Reference group for the ${this.node.id} service`,
        vpc: this.vpc,
      });

      this.securityGroupIds = [this.securityGroup!.securityGroupId];
    }
  }
}
