package main

import (
	"log"
	"os"

	grafana "code.justin.tv/live/autohost-grafana"
)

func main() {
	args := os.Args
	if len(args) != 2 {
		log.Fatalf(`Usage:
    ./generate-autohost-server-dashboard OUT_FILE_PATH
`)
	}
	outputFilePath := args[1]

	err := Run(outputFilePath)
	if err != nil {
		log.Fatalln(err)
	}
}

func Run(outputFilePath string) error {
	clueHost := grafana.Dependency{
		Name: "twitchclient/Clue:Host",
	}
	clueUnhost := grafana.Dependency{
		Name: "twitchclient/Clue:Unhost",
	}
	dynamoDBGetItem := grafana.Dependency{
		Name: "dynamodb:GetItem",
	}
	dynamoDBDeleteItem := grafana.Dependency{
		Name: "dynamodb:DeleteItem",
	}
	dynamoDBUpdateItem := grafana.Dependency{
		Name: "dynamodb:UpdateItem",
	}
	hallpassGetV1IsEditor := grafana.Dependency{
		Name: "twitchclient/Hallpass:GetV1IsEditor",
	}
	livelineGetStreamsByChannelID := grafana.Dependency{
		Name: "liveline/Liveline:GetStreamsByChannelIDs",
	}
	redisPrimaryEval := grafana.Dependency{
		Name: "redisPrimary:eval",
	}
	snsPublish := grafana.Dependency{
		Name: "sns:Publish",
	}
	twitchRecsGetRecommendedStreams := grafana.Dependency{
		Name: "twitch.recs.twitchrecs/TwitchRecs:GetRecommendedStreams",
	}
	twitchs2s2GetAccessToken := grafana.Dependency{
		Name: "twitchs2s2:GetAccessToken",
	}
	twitchs2s2GetAssertion := grafana.Dependency{
		Name: "twitchs2s2:GetAssertion",
	}
	twitchs2s2GetClientCredentials := grafana.Dependency{
		Name: "twitchs2s2:GetClientCredentials",
	}
	usersGetUserByID := grafana.Dependency{
		Name: "twitchclient/Users:GetUserByID",
	}

	eventBusOperations := []grafana.Operation{
		{
			Name: "HandleUserDestroyEvent",
			Dependencies: []grafana.Dependency{
				dynamoDBDeleteItem,
				dynamoDBUpdateItem,
			},
		},
	}
	hostingLegacyOps := []grafana.Operation{
		{
			Name: "api/GetSettings",
			Dependencies: []grafana.Dependency{
				dynamoDBGetItem,
			},
		}, {
			Name: "api/HostTarget",
			Dependencies: []grafana.Dependency{
				redisPrimaryEval,
				clueHost,
				hallpassGetV1IsEditor,
				usersGetUserByID,
			},
		}, {
			Name: "api/Unhost",
			Dependencies: []grafana.Dependency{
				redisPrimaryEval,
				clueUnhost,
				hallpassGetV1IsEditor,
				usersGetUserByID,
			},
		}, {
			Name: "api/UpdateSettings",
			Dependencies: []grafana.Dependency{
				dynamoDBUpdateItem,
			},
		},
	}
	hostingTwirpOperations := []grafana.Operation{
		{
			Name: "AddToAutohostList",
			Dependencies: []grafana.Dependency{
				snsPublish,
				dynamoDBGetItem,
				dynamoDBUpdateItem,
			},
		}, {
			Name: "GetAutohostList",
			Dependencies: []grafana.Dependency{
				dynamoDBGetItem,
			},
		}, {
			Name: "GetAutohostChannelForCarousel",
			Dependencies: []grafana.Dependency{
				dynamoDBGetItem,
				livelineGetStreamsByChannelID,
				twitchRecsGetRecommendedStreams,
				twitchs2s2GetAccessToken,
				twitchs2s2GetAssertion,
				twitchs2s2GetClientCredentials,
			},
		}, {
			Name: "GetAutohostSettings",
			Dependencies: []grafana.Dependency{
				dynamoDBGetItem,
			},
		}, {
			Name: "GetEndorsedChannels",
			Dependencies: []grafana.Dependency{
				dynamoDBGetItem,
				livelineGetStreamsByChannelID,
				twitchRecsGetRecommendedStreams,
				twitchs2s2GetAccessToken,
				twitchs2s2GetAssertion,
				twitchs2s2GetClientCredentials,
			},
		}, {
			Name: "RemoveFromAutohostList",
			Dependencies: []grafana.Dependency{
				dynamoDBGetItem,
				dynamoDBUpdateItem,
				snsPublish,
			},
		}, {
			Name: "SetAutohostList",
			Dependencies: []grafana.Dependency{
				dynamoDBGetItem,
				dynamoDBUpdateItem,
				snsPublish,
				hallpassGetV1IsEditor,
			},
		}, {
			Name: "UpdateAutohostSettings",
			Dependencies: []grafana.Dependency{
				dynamoDBUpdateItem,
			},
		},
	}

	builder := grafana.NewDashboardBuilder(&grafana.DashboardBuilderArgs{
		Namespace:       "Autohost Server",
		Service:         "Autohost Server",
		DashboardID:     439,
		DashboardTitle:  "Autohost Server",
		DashboardUID:    "jb-P7osZz",
		DataSourceRegex: "/^twitch-ce-host/",
		Stages:          []string{"production", "staging", "development"},
		SubStages:       []string{"primary"},
	})
	builder.AddTitlePanel("Overview")
	builder.AddRow("Errors Overview", false)
	builder.AddErrorSummaryPanels(hostingLegacyOps, "Hosting Legacy APIs")
	builder.AddErrorSummaryPanels(hostingTwirpOperations, "Hosting Twirp APIs")
	builder.AddErrorSummaryPanels(eventBusOperations, "Event Bus Handlers")

	latencyOverviewRowID := builder.AddRow("Latency Overview", true)
	builder.AddLatencySummaryPanels(hostingLegacyOps, latencyOverviewRowID, "Hosting Legacy APIs")
	builder.AddLatencySummaryPanels(hostingTwirpOperations, latencyOverviewRowID, "Hosting Twirp APIs")
	builder.AddLatencySummaryPanels(eventBusOperations, latencyOverviewRowID, "Event Bus Handlers")

	throughputOverviewRowID := builder.AddRow("Throughput Overview", true)
	builder.AddThroughputSummaryPanels(hostingLegacyOps, throughputOverviewRowID, "Hosting Legacy APIs")
	builder.AddThroughputSummaryPanels(hostingTwirpOperations, throughputOverviewRowID, "Hosting Twirp APIs")
	builder.AddThroughputSummaryPanels(eventBusOperations, throughputOverviewRowID, "Event Bus Handlers")

	builder.AddDetailPanels(hostingLegacyOps, "Hosting Legacy API Details")
	builder.AddDetailPanels(hostingTwirpOperations, "Hosting Twirp API Details")
	builder.AddDetailPanels(eventBusOperations, "Event Bus Handler Details")

	return builder.ToFile(outputFilePath)
}
