package main

import (
	"log"
	"os"

	grafana "code.justin.tv/live/autohost-grafana"
)

func main() {
	args := os.Args
	if len(args) != 2 {
		log.Fatalf(`Usage:
    ./generate-nexus-dashboard OUT_FILE_PATH
`)
	}
	outputFilePath := args[1]

	err := Run(outputFilePath)
	if err != nil {
		log.Fatalln(err)
	}
}

func Run(outputFilePath string) error {
	categoryDBGetGameList := grafana.Dependency{
		Name:        "CategoryDB:GetGameList",
		CircuitName: "CategoryDBClient_ValidateCategoryIDs",
	}
	dynamoDBDeleteItem := grafana.Dependency{
		Name: "dynamodb:DeleteItem",
	}
	dynamoDBGetItem := grafana.Dependency{
		Name: "dynamodb:GetItem",
	}
	dynamoDBPutItem := grafana.Dependency{
		Name: "dynamodb:PutItem",
	}
	dynamoDBQuery := grafana.Dependency{
		Name: "dynamodb:Query",
	}
	hallpassGetV1PermissionsChannelsEditors := grafana.Dependency{
		Name:        "Hallpass:GetV1PermissionsChannelsEditors",
		CircuitName: "HallpassClient_IsChannelEditor",
	}
	redisDel := grafana.Dependency{
		Name: "redis:del",
	}
	redisGet := grafana.Dependency{
		Name: "redis:get",
	}
	redisSet := grafana.Dependency{
		Name: "redis:set",
	}
	zumaPostV2MaapMessagesEnforce := grafana.Dependency{
		Name:        "Zuma:PostV2MaapMessagesEnforce",
		CircuitName: "ZumaClient_Moderate",
	}

	dependencies := []grafana.Dependency{
		categoryDBGetGameList,
		dynamoDBDeleteItem,
		dynamoDBGetItem,
		dynamoDBPutItem,
		dynamoDBQuery,
		hallpassGetV1PermissionsChannelsEditors,
		redisDel,
		redisGet,
		redisSet,
		zumaPostV2MaapMessagesEnforce,
	}

	operations := []grafana.Operation{
		{
			Name: "BatchGetNextSegment",
			Dependencies: []grafana.Dependency{
				dynamoDBGetItem,
				dynamoDBQuery,
				redisGet,
				redisSet,
			},
		},
		{
			Name: "BatchGetSegmentList",
			Dependencies: []grafana.Dependency{
				dynamoDBQuery,
				redisGet,
				redisSet,
			},
		},
		{
			Name: "BatchGetSchedule",
			Dependencies: []grafana.Dependency{
				dynamoDBGetItem,
				redisGet,
				redisSet,
			},
		},
		{
			Name: "BatchGetSegments",
			Dependencies: []grafana.Dependency{
				dynamoDBQuery,
				redisGet,
				redisSet,
			},
		},
		{
			Name: "CreateSchedule",
			Dependencies: []grafana.Dependency{
				hallpassGetV1PermissionsChannelsEditors,
				dynamoDBGetItem,
				dynamoDBPutItem,
				redisDel,
				redisGet,
				redisSet,
			},
		},
		{
			Name: "CreateSegment",
			Dependencies: []grafana.Dependency{
				categoryDBGetGameList,
				dynamoDBGetItem,
				dynamoDBPutItem,
				dynamoDBQuery,
				hallpassGetV1PermissionsChannelsEditors,
				redisDel,
				redisGet,
				redisSet,
				zumaPostV2MaapMessagesEnforce,
			},
		},
		{
			Name: "DeleteSchedule",
			Dependencies: []grafana.Dependency{
				hallpassGetV1PermissionsChannelsEditors,
				dynamoDBDeleteItem,
				dynamoDBQuery,
				redisDel,
			},
		},
		{
			Name: "DeleteSegment",
			Dependencies: []grafana.Dependency{
				hallpassGetV1PermissionsChannelsEditors,
				dynamoDBDeleteItem,
				redisDel,
			},
		},
		{
			Name:         "EventbusUserDestroy",
			Dependencies: []grafana.Dependency{},
		},
		{
			Name: "GetSchedule",
			Dependencies: []grafana.Dependency{
				dynamoDBGetItem,
				redisGet,
				redisSet,
			},
		},
		{
			Name: "GetSegment",
			Dependencies: []grafana.Dependency{
				dynamoDBQuery,
			},
		},
		{
			Name: "GetSegments",
			Dependencies: []grafana.Dependency{
				dynamoDBQuery,
				redisGet,
				redisSet,
			},
		},
		{
			Name: "InterruptSchedule",
			Dependencies: []grafana.Dependency{
				hallpassGetV1PermissionsChannelsEditors,
				dynamoDBGetItem,
				dynamoDBPutItem,
				redisDel,
			},
		},
		{
			Name: "ResumeSchedule",
			Dependencies: []grafana.Dependency{
				hallpassGetV1PermissionsChannelsEditors,
				dynamoDBGetItem,
				dynamoDBPutItem,
				redisDel,
			},
		},
		{
			Name: "UpdateSegment",
			Dependencies: []grafana.Dependency{
				categoryDBGetGameList,
				hallpassGetV1PermissionsChannelsEditors,
				zumaPostV2MaapMessagesEnforce,
				dynamoDBGetItem,
				dynamoDBPutItem,
				dynamoDBQuery,
				redisDel,
				redisGet,
				redisSet,
			},
		},
	}

	builder := grafana.NewDashboardBuilder(&grafana.DashboardBuilderArgs{
		Namespace:       "StreamSchedule",
		Service:         "StreamSchedule",
		DashboardID:     1734,
		DashboardTitle:  "StreamSchedule - API",
		DashboardUID:    "erMYmKvGk",
		DataSourceRegex: "/^streamschedule/",
		Stages:          []string{"prod", "beta"},
		SubStages:       []string{"primary", "canary"},
	})
	builder.AddTitlePanel("Overview")
	builder.AddRow("Errors Overview", false)
	builder.AddErrorSummaryPanels(operations, "Twirp APIs")

	latencyOverviewRowID := builder.AddRow("Latency Overview", true)
	builder.AddLatencySummaryPanels(operations, latencyOverviewRowID, "Twirp APIs")

	throughputOverviewRowID := builder.AddRow("Throughput Overview", true)
	builder.AddThroughputSummaryPanels(operations, throughputOverviewRowID, "Twirp APIs")

	builder.AddDependencyDetailPanels(dependencies)
	builder.AddDetailPanels(operations, "API Details")

	return builder.ToFile(outputFilePath)
}
