#!/usr/bin/env node
import "source-map-support/register";
import * as cdk from "@aws-cdk/core";
import { MetricsStack } from "../lib/stacks/metrics";
import { cacheM5Large, RedisStack } from "../lib/stacks/redis";
import * as accountID from "../lib/account-ids";
import { EnvName } from "../lib/env-names";
import * as network from "../lib/network";
import { MemcachedStack } from "../lib/stacks/memcached";
import { RaidsStack } from "../lib/stacks/raids";
import { ServiceInfraStack } from "../lib/stacks/service-infra";
import { VPCInfraStack } from "../lib/stacks/vpc-infra";

const stagingEnv: cdk.Environment = {
  account: accountID.twitchCEHostDev,
  region: "us-west-2",
};
const productionEnv: cdk.Environment = {
  account: accountID.twitchCEHostAWS,
  region: "us-west-2",
};

const app = new cdk.App();

const stagingMetrics = new MetricsStack(app, "StagingMetricsStack", {
  env: stagingEnv,
  envName: EnvName.Staging,
  vpcID: "vpc-051ac9d6a1ba4233c",
});

const prodMetrics = new MetricsStack(app, "ProductionMetricsStack", {
  env: productionEnv,
  envName: EnvName.Production,
  vpcID: "vpc-0d6b0519d6ee35100",
});

new RedisStack(app, "StagingRaidsRedisStack", {
  env: stagingEnv,
  alarmTwitchSeverity: 3,
  cloudwatchAlarmSNSTopics: stagingMetrics.cloudwatchAlarmSNSTopics,
  clusterDesc: "Raids staging Redis cluster",
  envName: EnvName.Staging,
  nodeType: cacheM5Large,
  numShards: 1,
  numNodesPerShard: 2,
  securityGroupIDs: [network.twitchSubnetsStagingSecurityGroupID],
  shardNamePrefix: "raids",
});

new RedisStack(app, "ProductionRaidsRedisStack", {
  env: productionEnv,
  alarmTwitchSeverity: 2.5,
  cloudwatchAlarmSNSTopics: prodMetrics.cloudwatchAlarmSNSTopics,
  clusterDesc: "Production Raids Redis cluster",
  envName: EnvName.Production,
  nodeType: cacheM5Large,
  numShards: 1,
  numNodesPerShard: 3,
  securityGroupIDs: [network.twitchSubnetsProductionSecurityGroupID],
  shardNamePrefix: "raids",
});

new MemcachedStack(app, "StagingMemcachedStack", {
  env: stagingEnv,
  envName: EnvName.Staging,
  clusterName: "aut-el-l88ehc0gq9qo",
  numNodes: 3,
  cloudwatchAlarmSNSTopics: stagingMetrics.cloudwatchAlarmSNSTopics,
  alarmTwitchSeverity: 3,
});

new MemcachedStack(app, "ProductionMemcachedStack", {
  env: productionEnv,
  envName: EnvName.Production,
  clusterName: "aut-el-1aw863r4xzf0j",
  numNodes: 3,
  cloudwatchAlarmSNSTopics: prodMetrics.cloudwatchAlarmSNSTopics,
  alarmTwitchSeverity: 2.5,
});

new RaidsStack(app, "StagingRaidsStack", {
  environmentName: EnvName.Staging,
  cloudwatchAlarmSNSTopics: stagingMetrics.cloudwatchAlarmSNSTopics,
});

new ServiceInfraStack(app, "ProductionServiceInfraStack", {
  description:
    "Production Autohost and hosting service related infrastructure. There may be multiple service-infra stacks in an AWS account.",
  env: productionEnv,
  envName: EnvName.Production,
  terminationProtection: true,
});

new ServiceInfraStack(app, "StagingServiceInfraStack", {
  description:
    "Staging Autohost and hosting service related infrastructure. There may be multiple service-infra stacks in an AWS account.",
  env: stagingEnv,
  envName: EnvName.Staging,
  terminationProtection: true,
});

new VPCInfraStack(app, "ProductionVPCInfraStack", {
  env: productionEnv,
  envName: EnvName.Production,
  terminationProtection: true,
  vpcID: "vpc-0d6b0519d6ee35100",
});

new VPCInfraStack(app, "StagingVPCInfraStack", {
  env: stagingEnv,
  envName: EnvName.Staging,
  terminationProtection: true,
  vpcID: "vpc-051ac9d6a1ba4233c",
});
