import * as cdk from "@aws-cdk/core";
import { EnvName } from "../../env-names";
import { CfnSubscriptionFilter } from "@aws-cdk/aws-logs";

interface Params {
  // Defines the log groups that subscriptions should be created for.
  readonly importLogGroupsProps: ImportLogGroupProps[];

  // The ARN of the TwitchSecurityLogFunnel's Kinesis Firehose delivery stream.
  // The subscription writes logs from the log groups to this firehose stream, and firehose
  // sends the logs to Security's infrastructure.
  readonly firehoseDeliveryStreamARN: string;

  // The IAM role for CloudWatch Logs to use when writing to the Kinesis Firehose delivery stream.
  readonly subscriptionRoleARN: string;
}

interface ImportLogGroupProps {
  // A string that can be used to build a unique construct ID.
  readonly constructIDSuffix: string;

  // The log group that subscription should be created for.
  readonly logGroupName: string;
}

const envToParams: Record<EnvName, Params> = {
  [EnvName.Production]: {
    importLogGroupsProps: [
      {
        constructIDSuffix: "WorkerDocker",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-production-worker/var/log/docker",
      },
      {
        constructIDSuffix: "WorkerDockerEvents",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-production-worker/var/log/docker-events.log",
      },
      {
        constructIDSuffix: "WorkerEBActivity",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-production-worker/var/log/eb-activity.log",
      },
      {
        constructIDSuffix: "WorkerService",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-production-worker/var/log/eb-docker/containers/eb-current-app/stdouterr.log",
      },
      {
        constructIDSuffix: "WorkerNginxAccess",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-production-worker/var/log/nginx/access.log",
      },
      {
        constructIDSuffix: "WorkerNginxError",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-production-worker/var/log/nginx/error.log",
      },
      {
        constructIDSuffix: "ServerDocker",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-server-production/var/log/docker",
      },
      {
        constructIDSuffix: "ServerDockerEvents",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-server-production/var/log/docker-events.log",
      },
      {
        constructIDSuffix: "ServerEBActivity",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-server-production/var/log/eb-activity.log",
      },
      {
        constructIDSuffix: "ServerService",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-server-production/var/log/eb-docker/containers/eb-current-app/stdouterr.log",
      },
      {
        constructIDSuffix: "ServerNginxAccess",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-server-production/var/log/nginx/access.log",
      },
      {
        constructIDSuffix: "ServerNginxError",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-server-production/var/log/nginx/error.log",
      },
    ],
    firehoseDeliveryStreamARN:
      "arn:aws:firehose:us-west-2:447680546588:deliverystream/twitch-security-log-funnel-FirehoseDeliveryStream-HEnTbOhcJmNc",
    subscriptionRoleARN:
      "arn:aws:iam::447680546588:role/twitch-security-log-funnel-CWLRole-EIC2PLJFGBYC",
  },
  [EnvName.Staging]: {
    importLogGroupsProps: [
      {
        constructIDSuffix: "WorkerDocker",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-development-worker/var/log/docker",
      },
      {
        constructIDSuffix: "WorkerDockerEvents",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-development-worker/var/log/docker-events.log",
      },
      {
        constructIDSuffix: "WorkerEBActivity",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-development-worker/var/log/eb-activity.log",
      },
      {
        constructIDSuffix: "WorkerService",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-development-worker/var/log/eb-docker/containers/eb-current-app/stdouterr.log",
      },
      {
        constructIDSuffix: "WorkerNginxAccess",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-development-worker/var/log/nginx/access.log",
      },
      {
        constructIDSuffix: "WorkerNginxError",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-development-worker/var/log/nginx/error.log",
      },
      {
        constructIDSuffix: "ServerDocker",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-server-staging/var/log/docker-events.log",
      },
      {
        constructIDSuffix: "ServerEBActivity",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-server-staging/var/log/eb-activity.log",
      },
      {
        constructIDSuffix: "ServerService",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-server-staging/var/log/eb-docker/containers/eb-current-app/stdouterr.log",
      },
      {
        constructIDSuffix: "ServerNginxAccess",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-server-staging/var/log/nginx/access.log",
      },
      {
        constructIDSuffix: "ServerNginxError",
        logGroupName:
          "/aws/elasticbeanstalk/autohost-server-staging/var/log/nginx/error.log",
      },
    ],
    firehoseDeliveryStreamARN:
      "arn:aws:firehose:us-west-2:038504843107:deliverystream/twitch-security-log-funnel-FirehoseDeliveryStream-lmTrWZFViasM",
    subscriptionRoleARN:
      "arn:aws:iam::038504843107:role/twitch-security-log-funnel-CWLRole-ASXEJU7L72HT",
  },
};

export interface TwitchSecurityLogFunnelSubscriptionsProps {
  readonly envName: EnvName;
}

// TwitchSecurityLogFunnelSubscriptions creates subscriptions between a set of log groups
// and the Twitch Security log funnel that we've created in our account. The log funnel
// The subscription sends our logs to the log funnel, and the funnel sends those logs to
// Security's infrastructure.
export class TwitchSecurityLogFunnelSubscriptions extends cdk.Construct {
  constructor(
    scope: cdk.Construct,
    id: string,
    props: TwitchSecurityLogFunnelSubscriptionsProps
  ) {
    super(scope, id);

    const {
      importLogGroupsProps,
      firehoseDeliveryStreamARN,
      subscriptionRoleARN,
    } = envToParams[props.envName];

    for (const { constructIDSuffix, logGroupName } of importLogGroupsProps) {
      new CfnSubscriptionFilter(
        this,
        `SubscriptionFilter${constructIDSuffix}`,
        {
          destinationArn: firehoseDeliveryStreamARN,
          filterPattern: "",
          logGroupName,
          roleArn: subscriptionRoleARN,
        }
      );
    }
  }
}
