package logic

import (
	"testing"

	"github.com/stretchr/testify/require"
)

func TestListDiff(t *testing.T) {
	t.Run("empty new list", func(t *testing.T) {
		added, removed := listDiff([]string{}, []string{"b", "a"})

		require.Empty(t, added)
		require.Len(t, removed, 2)
		require.Contains(t, removed, "b")
		require.Contains(t, removed, "a")
	})

	t.Run("empty old list", func(t *testing.T) {
		added, removed := listDiff([]string{"a", "b"}, []string{})

		require.Len(t, added, 2)
		require.Contains(t, added, "a")
		require.Contains(t, added, "b")
		require.Empty(t, removed)
	})

	t.Run("empty input lists", func(t *testing.T) {
		added, removed := listDiff([]string{}, []string{})

		require.Empty(t, added)
		require.Empty(t, removed)
	})

	t.Run("no diff if no diff", func(t *testing.T) {
		added, removed := listDiff([]string{"a", "b"}, []string{"b", "a"})

		require.Empty(t, added)
		require.Empty(t, removed)
	})

	t.Run("old list has more elements", func(t *testing.T) {
		added, removed := listDiff([]string{"a", "b"}, []string{"b", "d", "a", "c"})

		require.Empty(t, added)
		require.Len(t, removed, 2)
		require.Contains(t, removed, "d")
		require.Contains(t, removed, "c")
	})

	t.Run("list has new duplicate elements", func(t *testing.T) {
		added, removed := listDiff([]string{"a", "b", "c", "c"}, []string{"a", "d", "e", "e"})

		require.Len(t, added, 2)
		require.Contains(t, added, "b")
		require.Contains(t, added, "c")

		require.Len(t, removed, 2)
		require.Contains(t, removed, "d")
		require.Contains(t, removed, "e")
	})

	t.Run("list has new elements", func(t *testing.T) {
		added, removed := listDiff([]string{"a", "b", "c"}, []string{"b", "d", "a"})

		require.Equal(t, []string{"c"}, added)
		require.Equal(t, []string{"d"}, removed)
	})
}

func TestRemoveElementsFromList(t *testing.T) {
	t.Run("Return empty list when input list is empty", func(t *testing.T) {
		newList := removeElementsFromList([]string{}, []string{"b", "a", "c", "e"})

		require.Empty(t, newList)
	})

	t.Run("Return same list when itemsToRemove list is empty", func(t *testing.T) {
		oldList := []string{"b", "a", "c", "e"}
		newList := removeElementsFromList(oldList, []string{})

		require.Equal(t, oldList, newList)
	})

	t.Run("Return new list with items removed", func(t *testing.T) {
		oldList := []string{"b", "a", "c", "d"}
		newList := removeElementsFromList(oldList, []string{"b", "c"})

		require.Equal(t, []string{"a", "d"}, newList)
	})

	t.Run("Remove duplicate in itemsToRemove from list", func(t *testing.T) {
		oldList := []string{"b", "a", "c", "d", "c"}
		newList := removeElementsFromList(oldList, []string{"b", "c", "b"})

		require.Equal(t, []string{"a", "d"}, newList)
	})
}
