// +build integration

package logic

import (
	"context"
	"testing"

	"code.justin.tv/live/autohost/internal/hosting/auth"
	"github.com/stretchr/testify/require"
)

func TestRemoveList(t *testing.T) {
	t.Run("Removes users from list", func(t *testing.T) {
		channelID := NewRandomUserID()
		callerID := NewRandomUserID()
		auth_ := auth.NewAuthStub().WithCanEditChannel(callerID, channelID)

		testEnv := newTestEnv(t, &Params{
			Auth: auth_,
		})
		l := testEnv.logic

		user1 := NewRandomUserID()
		user2 := NewRandomUserID()
		user3 := NewRandomUserID()
		originalList := []string{user1, user2, user3}

		_, err := l.AddList(context.Background(), callerID, channelID, originalList)
		require.NoError(t, err)

		user4 := NewRandomUserID()
		usersToRemove := []string{user2, user3, user4}

		updatedList, err := l.RemoveList(context.Background(), callerID, channelID, usersToRemove)
		require.NoError(t, err)
		require.Len(t, updatedList, 1)
		require.Contains(t, updatedList, user1)
	})

	t.Run("Returns error when not authorized", func(t *testing.T) {
		channelID := NewRandomUserID()
		callerID := NewRandomUserID()
		auth_ := auth.NewAuthStub().WithCanEditChannel(callerID, channelID)

		testEnv := newTestEnv(t, &Params{
			Auth: auth_,
		})
		l := testEnv.logic

		user1 := NewRandomUserID()
		user2 := NewRandomUserID()
		user3 := NewRandomUserID()
		originalList := []string{user1, user2, user3}

		_, err := l.AddList(context.Background(), callerID, channelID, originalList)
		require.NoError(t, err)

		callerID2 := NewRandomUserID()
		usersToRemove := []string{user2}
		_, err = l.RemoveList(context.Background(), callerID2, channelID, usersToRemove)
		require.Error(t, err)
	})

	t.Run("Doesn't do anything if list is empty", func(t *testing.T) {
		channelID := NewRandomUserID()
		callerID := NewRandomUserID()
		auth_ := auth.NewAuthStub().WithCanEditChannel(callerID, channelID)

		testEnv := newTestEnv(t, &Params{
			Auth: auth_,
		})
		l := testEnv.logic

		user1 := NewRandomUserID()
		usersToRemove := []string{user1}

		updatedList, err := l.RemoveList(context.Background(), callerID, channelID, usersToRemove)
		require.NoError(t, err)
		require.Empty(t, updatedList)
	})
}
