package server

import (
	"context"
	"fmt"
	"testing"

	"code.justin.tv/live/autohost/rpc/hosting"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"
)

func TestAddToAutohostList_Success(t *testing.T) {
	testEnv := newTestEnvWithMockLogic(t)
	mockLogic := testEnv.mockLogic
	client := testEnv.client

	callerID := "11111111"
	userID := "222222222"
	channelsToAdd := []string{"33333333", "44444444"}
	expectedUpdatedList := []string{"55555555", "66666666", "33333333", "44444444"}
	mockLogic.
		On("AddList", mock.Anything, callerID, userID, channelsToAdd).
		Return(expectedUpdatedList, nil).Once()

	ctx := context.Background()
	resp, err := client.AddToAutohostList(ctx, &hosting.AddToAutohostListRequest{
		CallerId:       callerID,
		SourceId:       userID,
		TargetIdsToAdd: channelsToAdd,
	})
	require.NoError(t, err)
	require.NotNil(t, resp)

	expectedResponse := &hosting.AddToAutohostListResponse{
		TargetIds: expectedUpdatedList,
	}
	require.Equal(t, expectedResponse, resp)
}

func TestAddToAutohostList_ServerError(t *testing.T) {
	testEnv := newTestEnvWithMockLogic(t)
	mockLogic := testEnv.mockLogic
	client := testEnv.client

	callerID := "11111111"
	userID := "222222222"
	channelsToAdd := []string{"33333333", "44444444"}
	err := fmt.Errorf("a test error")
	mockLogic.
		On("AddList", mock.Anything, callerID, userID, channelsToAdd).
		Return(nil, err).Once()

	ctx := context.Background()
	resp, err := client.AddToAutohostList(ctx, &hosting.AddToAutohostListRequest{
		CallerId:       callerID,
		SourceId:       userID,
		TargetIdsToAdd: channelsToAdd,
	})
	requireTwirpError(t, twirp.Internal, err)
	require.Nil(t, resp)
}
