package server

import (
	"context"
	"fmt"
	"testing"

	"code.justin.tv/live/autohost/rpc/hosting"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"
)

func TestRemoveFromAutohostList_Success(t *testing.T) {
	testEnv := newTestEnvWithMockLogic(t)
	mockLogic := testEnv.mockLogic
	client := testEnv.client

	callerID := "11111111"
	userID := "222222222"
	channelsToRemove := []string{"33333333", "44444444"}
	expectedUpdatedList := []string{"55555555", "66666666"}
	mockLogic.
		On("RemoveList", mock.Anything, callerID, userID, channelsToRemove).
		Return(expectedUpdatedList, nil).Once()

	ctx := context.Background()
	resp, err := client.RemoveFromAutohostList(ctx, &hosting.RemoveFromAutohostListRequest{
		CallerId:          callerID,
		SourceId:          userID,
		TargetIdsToRemove: channelsToRemove,
	})
	require.NoError(t, err)
	require.NotNil(t, resp)

	expectedResponse := &hosting.RemoveFromAutohostListResponse{
		TargetIds: expectedUpdatedList,
	}
	require.Equal(t, expectedResponse, resp)
}

func TestRemoveFromAutohostList_ServerError(t *testing.T) {
	testEnv := newTestEnvWithMockLogic(t)
	mockLogic := testEnv.mockLogic
	client := testEnv.client

	callerID := "11111111"
	userID := "222222222"
	channelsToRemove := []string{"33333333", "44444444"}
	err := fmt.Errorf("a test error")
	mockLogic.
		On("RemoveList", mock.Anything, callerID, userID, channelsToRemove).
		Return(nil, err).Once()

	ctx := context.Background()
	resp, err := client.RemoveFromAutohostList(ctx, &hosting.RemoveFromAutohostListRequest{
		CallerId:          callerID,
		SourceId:          userID,
		TargetIdsToRemove: channelsToRemove,
	})
	requireTwirpError(t, twirp.Internal, err)
	require.Nil(t, resp)
}
