package server

import (
	"testing"

	"github.com/golang/protobuf/ptypes/wrappers"

	"code.justin.tv/live/autohost/rpc/hosting"
	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"
)

func TestUpdateAutohostSettings_AllSettings(t *testing.T) {
	ctx, cancel := getContextWithTimeout()
	defer cancel()

	userID := "1234567"

	testEnv := newTestEnv(t)
	testEnv.authStub.AllowAllCalls = true
	hostingClient := testEnv.hostingClient

	resp, err := hostingClient.UpdateAutohostSettings(ctx, &hosting.UpdateAutohostSettingsRequest{
		CallerId: userID,
		UserId:   userID,
		Enabled: &wrappers.BoolValue{
			Value: true,
		},
		TeamHost: &wrappers.BoolValue{
			Value: true,
		},
		Strategy: hosting.AutohostStrategy_AUTOHOST_STRATEGY_ORDERED,
		DeprioritizeVodcast: &wrappers.BoolValue{
			Value: true,
		},
	})
	require.NoError(t, err)
	require.NotNil(t, resp)

	expectedResponse := &hosting.UpdateAutohostSettingsResponse{
		Settings: &hosting.AutohostSettings{
			Enabled:             true,
			TeamHost:            true,
			Strategy:            hosting.AutohostStrategy_AUTOHOST_STRATEGY_ORDERED,
			DeprioritizeVodcast: true,
		},
	}
	require.Equal(t, expectedResponse, resp)
}

func TestUpdateAutohostSettings_UpdateExistingSetting(t *testing.T) {
	ctx, cancel := getContextWithTimeout()
	defer cancel()

	userID := "1234567"

	testEnv := newTestEnv(t)
	testEnv.authStub.AllowAllCalls = true
	hostingClient := testEnv.hostingClient

	// Set initial settings.
	_, err := hostingClient.UpdateAutohostSettings(ctx, &hosting.UpdateAutohostSettingsRequest{
		CallerId: userID,
		UserId:   userID,
		Enabled: &wrappers.BoolValue{
			Value: true,
		},
		TeamHost: &wrappers.BoolValue{
			Value: true,
		},
		Strategy: hosting.AutohostStrategy_AUTOHOST_STRATEGY_ORDERED,
		DeprioritizeVodcast: &wrappers.BoolValue{
			Value: true,
		},
	})
	require.NoError(t, err)

	// Update some fields.
	resp, err := hostingClient.UpdateAutohostSettings(ctx, &hosting.UpdateAutohostSettingsRequest{
		CallerId: userID,
		UserId:   userID,
		Enabled: &wrappers.BoolValue{
			Value: false,
		},
		Strategy: hosting.AutohostStrategy_AUTOHOST_STRATEGY_RANDOM,
	})
	require.NoError(t, err)
	require.NotNil(t, resp)

	// Verify that the fields were updated.
	expectedResponse := &hosting.UpdateAutohostSettingsResponse{
		Settings: &hosting.AutohostSettings{
			Enabled:             false,
			TeamHost:            true,
			Strategy:            hosting.AutohostStrategy_AUTOHOST_STRATEGY_RANDOM,
			DeprioritizeVodcast: true,
		},
	}
	require.Equal(t, expectedResponse, resp)
}

func TestUpdateAutohostSettings_Defaults(t *testing.T) {
	ctx, cancel := getContextWithTimeout()
	defer cancel()

	userID := "1234567"

	testEnv := newTestEnv(t)
	testEnv.authStub.AllowAllCalls = true
	hostingClient := testEnv.hostingClient

	resp, err := hostingClient.UpdateAutohostSettings(ctx, &hosting.UpdateAutohostSettingsRequest{
		CallerId: userID,
		UserId:   userID,
	})
	require.NoError(t, err)
	require.NotNil(t, resp)

	expectedResponse := &hosting.UpdateAutohostSettingsResponse{
		Settings: &hosting.AutohostSettings{
			Enabled:             false,
			TeamHost:            false,
			Strategy:            hosting.AutohostStrategy_AUTOHOST_STRATEGY_RANDOM,
			DeprioritizeVodcast: false,
		},
	}
	require.Equal(t, expectedResponse, resp)
}

func TestUpdateAutohostSettings_Unauthorized(t *testing.T) {
	ctx, cancel := getContextWithTimeout()
	defer cancel()

	userID := "1234567"

	testEnv := newTestEnv(t)
	testEnv.authStub.AllowAllCalls = false
	hostingClient := testEnv.hostingClient

	resp, err := hostingClient.UpdateAutohostSettings(ctx, &hosting.UpdateAutohostSettingsRequest{
		CallerId: userID,
		UserId:   userID,
	})
	requireTwirpError(t, twirp.PermissionDenied, err)
	require.Nil(t, resp)
}
